#pragma once

#include "Component.h"
#include "ListBox.h"

class AutoCompleteListener
{
public:

    virtual BOOL getSuggestions(const wstring& text, vector<wstring>& suggestions) = 0;
};

/**
 * @class AutoComplete
 * @brief A combo box UI component width auto-complete functionality.
 *
 * Combines a dropdown-style control with a listbox interface.
 * Inherits from Component and
 * implements SelectionChangeListener to react to internal selection events.
 */
class AutoComplete : public Component,
                      public SelectionChangeListener
{
    wstring _text;      /**< The current text content of the field */

    AutoCompleteListener* _listener;

    /**
     * @brief Internal listbox control used for item selection.
     */
    ListBox _listBox;

    void showList();
    void hideList();
    void toggleList();

protected:
    /**
     * @brief Retrieves window creation options for the control.
     *
     * @param title Output: window title.
     * @param widownStyles Output: window style flags.
     * @param wndClassName Output: class name of the window.
     * @param isCustomWndProc Output: whether a custom window procedure is used.
     * @return TRUE if successful, FALSE otherwise.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the control window has been successfully created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new AutoComplete instance.
     */
    AutoComplete();

    /**
     * @brief Destroys the AutoComplete instance.
     */
    virtual ~AutoComplete();

    /**
     * @brief Returns the preferred height
     */
    int getPreferredHeight();

    void setListener(AutoCompleteListener* l);

    /**
     * @brief Sets the text content of the field.
     *
     * @param text The text to set.
     */
    void setText(const wstring& text);

    /**
     * @brief Gets the current text content.
     *
     * @return The current text in the field.
     */
    wstring getText();

    /**
     * @brief Sets the available options that can be selected via the listbox.
     *
     * @param selections A list of key-value pairs representing selectable options.
     */
    void setOptions(const vector<wstring>& selections);

    /**
     * @brief Renders the AutoComplete.
     *
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Called when the window containing the control is resized.
     */
    void onWindowResized();

    /**
     * @brief Called when the window containing the control is moved.
     */
    void onWindowMoved();

    /**
     * @brief Responds to selection changes triggered by internal components like CheckList.
     *
     * @param ev The selection change event containing details of the change.
     */
    void onSelectionChanged(const SelectionChangeEvent& ev);

    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles data changed events.
     *
     * @param hTarget Handle to the target window.
     * @param newValue The new text value.
     * @param procParams Additional parameters.
     */
    void onDataChanged(WinHandle hTarget, const wstring& newValue, const ProcParams& procParams);

    /**
     * @brief Adds a listener for data changed events.
     *
     * @param l Pointer to the listener.
     */
    void addDataChangedListener(DataChangeListener* l);
};
