#pragma once

#include "Component.h"

/**
 * @class Button
 * @brief A clickable UI component representing a button with customizable text and type.
 * 
 * The Button class extends Component and supports mouse interactions,
 * allowing for text display and different button styles (e.g., primary, default, danger, success).
 */
class Button : public Component
{
    wstring _text;          ///< The text displayed on the button.
    ButtonType _buttonType;    ///< The style/type of the button (e.g., ButtonType::SUCCESS).
    BOOL _isLink;
    BOOL _showShadow;
    BOOL _disableGradient;
    COLORREF _bkColor;    ///< The background color of the button in case buttonType is CUSTOM.
    COLORREF _fgColor;    ///< The foreground color of the button in case buttonType is CUSTOM.
    BOOL _isClicked;        ///< Indicates if the button is currently clicked.
    IconSource _iconSource;
    Direction _iconPosition;  /**< The position of the icon in case a button displays an icon. */

protected:
    /**
     * @brief Prepares window creation options such as title, styles, and class name.
     * 
     * @param title The window title to set.
     * @param widownStyles Window style flags.
     * @param wndClassName The window class name.
     * @param isCustomWndProc Indicates if a custom window procedure is used.
     * @return TRUE if the options were set successfully, FALSE otherwise.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the window has been created to perform any initialization.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a Button object.
     */
    Button();

    /**
     * @brief Destructs the Button object and cleans up resources.
     */
    virtual ~Button();

    /**
     * @brief Sets the text displayed on the button.
     * 
     * @param text The new text for the button.
     */
    void setText(const wstring& text);
    wstring getText();

    void setHasShadow(BOOL hasShadow);
    void setDisableGradient(BOOL disabled);

    /**
     * @brief Sets the image file displayed on the button and the image position.
     *
     * @param imageFile The file containing the image displayed on the button.
     * @param imagePosition The position of the image displayed on the button (e.g. left|top|right|bottom).
     */
    void setImage(const IconSource& iconSource, const Direction& imagePosition = Direction::LEFT);

    /**
     * @brief Sets the button type/style.
     * 
     * @param buttonType The type of button (e.g., ButtonType::PRIMARY).
     */
    void setButtonType(const ButtonType& buttonType,
        COLORREF bkColor = Theme::ButtonBackgroundStartColor, /* Will only be used is btnType is set to ButtonType::CUSTOM */
        COLORREF fgColor = Theme::ButtonForegroundColor /* Will only be used is btnType is set to ButtonType::CUSTOM */);

    void setLinkStyle();

    /**
     * @brief Handles the painting of the button.
     * 
     * @param hDC Handle to the device context to paint on.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Handles mouse press events on the button.
     * 
     * @param hWnd Handle to the window receiving the event.
     * @param x The x-coordinate of the mouse pointer.
     * @param y The y-coordinate of the mouse pointer.
     * @param clickCount Number of clicks detected.
     * @param shiftPressed TRUE if the Shift key is pressed.
     * @param ctrlPressed TRUE if the Ctrl key is pressed.
     */
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles mouse release events on the button.
     * 
     * @param hWnd Handle to the window receiving the event.
     * @param x The x-coordinate of the mouse pointer.
     * @param y The y-coordinate of the mouse pointer.
     * @param shiftPressed TRUE if the Shift key is pressed.
     * @param ctrlPressed TRUE if the Ctrl key is pressed.
     */
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles mouse move events over the button.
     * 
     * @param hWnd Handle to the window receiving the event.
     * @param x The x-coordinate of the mouse pointer.
     * @param y The y-coordinate of the mouse pointer.
     * @param shiftPressed TRUE if the Shift key is pressed.
     * @param ctrlPressed TRUE if the Ctrl key is pressed.
     */
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Adds an action listener to the button to handle action events.
     * 
     * @param l Pointer to the ActionListener object to add.
     */
    void addActionListener(ActionListener* l);
};
