#pragma once

#include "Component.h"
#include "Scrollbar.h"

class ButtonGridItem
{
public:
    IconSource _iconSource;
    wstring _text;
    BOOL _isDisabled;
    Bounds _rect, _imageRect, _textRect;
};

/**
 * @class ButtonGrid
 * @brief A scrollable, resizable grid component that displays buttons in a structured layout.
 *
 * The ButtonGrid class allows dynamic addition of buttons (with optional images), automatic layout
 * adjustment based on window size, and user interaction through mouse and keyboard. It supports vertical
 * scrolling, selection tracking, and event listeners for user actions.
 */
class ButtonGrid : public Component, public ScrollbarListener
{
    Scrollbar _scrollBar;

    wstring _title;                      /**< Title displayed above the grid. */
    vector<ButtonGridItem> _items;       /**< List of buttons in the grid. */
    Direction _iconPosition;               /**< The position of the icon in case a button displays an icon. */

    long _minWidth;                      /**< Minimum column width. */
    long _columnsPerRow;                 /**< Number of columns per row based on available space. */
    long _maxVisibleItems;               /**< Max number of visible items without scrolling. */
    __int64 _firstVisibleItem;              /**< Index of the first visible item (scrolling). */
    __int64 _lastVisibleItem;               /**< Index of the last visible item. */
    __int64 _currentItem;                   /**< Index of the currently selected/focused item. */

    __int64 _indexOfPressedItem;            /**< Index of the currently pressed item. */

protected:

    /**
     * @brief Sets the options used to create the native window.
     * @param[out] title Window title.
     * @param[out] windowStyles Window style flags.
     * @param[out] wndClassName Name of the window class.
     * @param[out] isCustomWndProc Whether a custom window procedure is used.
     * @return TRUE if options were successfully set.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& windowStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called when the window is created.
     */
    void windowCreated();

public:

    /** @brief Constructor. Initializes the grid. */
    ButtonGrid();

    /** @brief Destructor. Cleans up any allocated resources. */
    virtual ~ButtonGrid();

    /**
     * @brief Clears all items from the grid.
     */
    void clear();

    /**
     * @brief Sets the minimum width for columns in the grid layout.
     * @param minWidth The minimum width in pixels.
     */
    void setMinColumnWidth(long minWidth);

    /**
     * @brief Sets the grid's title.
     * @param title The title to display.
     */
    void setTitle(const wstring& title);

    /**
     * @brief Sets the position of the image displayed on the button.
     *
     * @param imagePosition The position of the image displayed on the button (e.g. left|top|right|bottom).
     */
    void setImagePosition(const Direction& imagePosition);

    /**
     * @brief Adds a button to the grid.
     * @param itemText The button's display text.
     * @param imageFile Optional image file path to show on the button.
     * @return The index of the added item.
     */
    long addButton(const wstring& itemText, const IconSource& iconSource = IconSource());

    /**
     * @brief enables/disabled button.
     *
     * @param itemText text of the item.
     * @param enabled TRUE to enable, FALSE to disable the item.
     */
    void enableButton(const wstring& itemText, BOOL enabled = TRUE);

    /**
     * @brief Handles window resize events and recalculates layout.
     */
    void onWindowResized();

    /**
     * @brief Paints the button grid on the screen.
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /** @name Keyboard Navigation */
    ///@{
    void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed);
    void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed);
    void onPageUp(BOOL shiftPressed, BOOL ctrlPressed);
    void onPageDown(BOOL shiftPressed, BOOL ctrlPressed);
    void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed);
    void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed);
    ///@}

    /** @name Mouse Events */
    ///@{
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseRightClicked(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);
    ///@}

    /**
     * @brief Registers an ActionListener to receive item click events.
     * @param l Pointer to the listener.
     */
    void addActionListener(ActionListener* l);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
