#pragma once

#include "Component.h"
#include "Scrollbar.h"

class ButtonMenuItem
{
public:
    wstring _id;
    wstring _text;
    IconSource _iconSource;
    BOOL _isDisabled;
    Bounds _rect, _imageRect, _textRect;
};

/**
 * @class ButtonMenu
 * @brief A scrollable, interactive list of selectable menu items with optional styling as buttons or links.
 *
 * The ButtonMenu class is a GUI component that displays a list of selectable items, supports keyboard and mouse
 * navigation, and notifies listeners when the selection changes. It is derived from the Component base class.
 */
class ButtonMenu : public Component, public ScrollbarListener
{
    Scrollbar _scrollBar;

    wstring _title;                       /**< Title of the menu. */
    vector<ButtonMenuItem> _items;        /**< Collection of menu items. */
    Direction _iconPosition;                /**< The position of the icon in case a button displays an icon. */
    BOOL _isPinnable;                     /**< Specifies whether the clicked button appears pinned. */

    long _maxVisibleItems;                /**< Maximum number of items visible without scrolling. */
    __int64 _firstVisibleItem;               /**< Index of the first visible item. */
    __int64 _lastVisibleItem;                /**< Index of the last visible item. */
    __int64 _currentItem;                    /**< Index of the currently selected item. */
    __int64 _moveOverItemIndex;              /**< Index of the item currently hovered by the mouse. */
    long _lastMouseMoveX;                    /**< x-coordinate of the last mouse move. */
    long _lastMouseMoveY;                    /**< y-coordinate of the last mouse move. */
    long _style;                          /**< Menu style: button or link. */

    __int64 _indexOfPressedItem;            /**< Index of the currently pressed item. */

protected:

    /**
     * @brief Retrieves window creation options.
     * @param[out] title The window title.
     * @param[out] windowStyles The style flags for the window.
     * @param[out] wndClassName The window class name.
     * @param[out] isCustomWndProc Indicates whether a custom window procedure is used.
     * @return TRUE if the options were set successfully.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& windowStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the window has been created.
     */
    void windowCreated();

public:

    /** @brief Constructor. */
    ButtonMenu();

    /** @brief Destructor. */
    virtual ~ButtonMenu();

    /** @brief Enables button-style rendering of items. */
    void enableButtonStyle();

    /** @brief Enables link-style rendering of items. */
    void enableLinkStyle();

    /** @brief Enables pinnable button state. */
    void setPinnable(BOOL isPinnable = TRUE);

    /** @brief Removes all items from the menu. */
    void clear();

    /**
     * @brief Sets the menu title.
     * @param title The new title.
     */
    void setTitle(const wstring& title);

    /**
     * @brief Sets the position of the image displayed on the button.
     *
     * @param imagePosition The position of the image displayed on the button (e.g. left|top|right|bottom).
     */
    void setImagePosition(const Direction& imagePosition);

    /**
     * @brief Adds a new item to the menu.
     * @param itemText Display text of the item.
     * @param itemId Optional identifier string.
     * @return Index of the newly added item.
     */
    long addItem(const wstring& itemText, const wstring& itemId = L"", const IconSource& iconSource = IconSource());

    /**
     * @brief enables/disabled button.
     *
     * @param itemText text of the item.
     * @param enabled TRUE to enable, FALSE to disable the item.
     */
    void enableItem(const wstring& itemText, BOOL enabled = TRUE);

    /** @brief Returns the number of items in the menu. */
    long getItemCount();

    /** @brief Gets the index of the currently selected item. */
    long getSelectedItem();

    /**
     * @brief Paints the component.
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /** @brief Called when the window is resized. */
    void onWindowResized();

    /** @name Keyboard Navigation */
    ///@{
    void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed);
    void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed);
    void onPageUp(BOOL shiftPressed, BOOL ctrlPressed);
    void onPageDown(BOOL shiftPressed, BOOL ctrlPressed);
    void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed);
    void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed);
    ///@}

    /** @name Mouse Interaction */
    ///@{
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseRightClicked(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);
    ///@}

    /**
     * @brief Adds a listener to be notified when the selection changes.
     * @param l Pointer to a SelectionChangeListener.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
