#pragma once

#include "Component.h"
#include "CheckList.h"

/**
 * @class CheckComboBox
 * @brief A combo box UI component that allows multiple selection using checkboxes.
 *
 * Combines a dropdown-style control with a checklist interface,
 * enabling users to select multiple items. Inherits from Component and
 * implements SelectionChangeListener to react to internal selection events.
 */
class CheckComboBox : public Component,
                      public SelectionChangeListener
{
    /**
     * @brief The text displayed when the dropdown is closed, summarizing selected items.
     */
    wstring _displayText;

    /**
     * @brief Internal checklist control used for item selection.
     */
    CheckList _checkList;

protected:
    /**
     * @brief Retrieves window creation options for the control.
     *
     * @param title Output: window title.
     * @param widownStyles Output: window style flags.
     * @param wndClassName Output: class name of the window.
     * @param isCustomWndProc Output: whether a custom window procedure is used.
     * @return TRUE if successful, FALSE otherwise.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the control window has been successfully created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new CheckComboBox instance.
     */
    CheckComboBox();

    /**
     * @brief Destroys the CheckComboBox instance.
     */
    virtual ~CheckComboBox();

    /**
     * @brief Returns the preferred height
     */
    int getPreferredHeight();

    /**
     * @brief Sets the available options that can be selected via the checklist.
     *
     * @param selections A list of key-value pairs representing selectable options.
     */
    void setOptions(const vector<KeyValue>& selections);

    /**
     * @brief Sets which items are selected based on a list of key-value pairs.
     *
     * @param selections A list of selected key-value pairs.
     */
    void setSelectedItems(const vector<KeyValue>& selections);

    /**
     * @brief Retrieves the list of currently selected items.
     *
     * @return A vector of selected key-value pairs.
     */
    vector<KeyValue> getSelectedItems();

    /**
     * @brief Renders the CheckComboBox.
     *
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Called when the window containing the control is resized.
     */
    void onWindowResized();

    /**
     * @brief Called when the window containing the control is moved.
     */
    void onWindowMoved();

    /**
     * @brief Handles mouse press events.
     *
     * @param hWnd Handle to the window receiving the event.
     * @param x X-coordinate of the mouse event.
     * @param y Y-coordinate of the mouse event.
     * @param clickCount Number of clicks (e.g., single or double).
     * @param shiftPressed Whether Shift was pressed during the click.
     * @param ctrlPressed Whether Ctrl was pressed during the click.
     */
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Responds to selection changes triggered by internal components like CheckList.
     *
     * @param ev The selection change event containing details of the change.
     */
    void onSelectionChanged(const SelectionChangeEvent& ev);

    /**
     * @brief Adds a listener to be notified when the selection changes.
     *
     * @param l Pointer to a SelectionChangeListener to notify.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);
};
