#pragma once

#include "Component.h"
#include "Scrollbar.h"

class CheckListItem
{
public:
    long _id;
    wstring _text;
    BOOL _checked;
    Bounds _checkRect;
};


/**
 * @class CheckList
 * @brief A UI component that displays a list of checkable items.
 *
 * CheckList allows for multi-selection using checkboxes. It can operate as a
 * standalone list or be embedded in popup-based components like CheckComboBox.
 */
class CheckList : public Component, public ScrollbarListener
{
    Scrollbar _scrollBar;

    /**
     * @brief The list of items currently in the checklist.
     */
    vector<CheckListItem> _items;

    /**
     * @brief A text representation of selected items (used internally).
     */
    wstring _selection;

    /**
     * @brief Text shown as a display summary (for popups).
     */
    wstring _displayText;

    /**
     * @brief Whether the checklist is shown as a popup.
     */
    BOOL _isPopup;

    long _maxVisibleRows;         ///< Maximum rows visible in the current view
    __int64 _firstVisibleRow;        ///< Index of the first visible row
    __int64 _lastVisibleRow;         ///< Index of the last visible row

protected:
    /**
     * @brief Retrieves the options for creating the checklist window.
     *
     * @param title Output: window title.
     * @param widownStyles Output: window style flags.
     * @param wndClassName Output: window class name.
     * @param isCustomWndProc Output: whether a custom window procedure is used.
     * @return TRUE if the options were successfully retrieved, FALSE otherwise.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the checklist window is created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new CheckList instance.
     */
    CheckList();

    /**
     * @brief Destroys the CheckList instance.
     */
    virtual ~CheckList();

    /**
     * @brief Sets whether the checklist is used in a popup context.
     *
     * @param isPopup TRUE to mark the checklist as a popup, FALSE otherwise.
     */
    void setAsPopup(BOOL isPopup);

    /**
     * @brief Return TRUE if this component is displayed as a popup (e.g. dropdown from CheckComboBox).
     */
    BOOL isPopup();

    /**
     * @brief Sets the available options to be shown in the checklist.
     *
     * @param selections A list of key-value pairs representing checklist items.
     */
    void setOptions(const vector<KeyValue>& selections);
    void getOptions(vector<KeyValue>& selections);

    /**
     * @brief Sets which items are selected based on key-value input.
     *
     * @param selections List of selected key-value items.
     */
    void setSelectedItems(const vector<KeyValue>& selections);

    /**
     * @brief Returns the currently selected items in the checklist.
     *
     * @return A vector of selected key-value pairs.
     */
    vector<KeyValue> getSelectedItems();

public: // Overridables

    void onPaint(Graphics* g) override;
    void onWindowResized() override;

    void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onPageUp(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onPageDown(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed) override;

    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed) override;
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta) override;

    void onFocusGained() override;;
    void onFocusLost() override;;

public:

    /**
     * @brief Notifies the checklist of a selection state change for an item.
     *
     * @param hWndCB Handle to the parent combo box or caller.
     * @param itemIndex Index of the item that changed.
     * @param itemText Text of the changed item.
     * @param itemChecked Whether the item is now checked.
     */
    void onSelectionChanged(WinHandle hWndCB, int itemIndex, const wstring& itemText, BOOL itemChecked);

    /**
     * @brief Adds a listener to be notified when item selection changes.
     *
     * @param l Pointer to a SelectionChangeListener instance.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
