#pragma once

#include "Common.h"
#include "GridPanel.h"

#define DIALOG_OK		1
#define DIALOG_CANCEL	0

// IMPORTANT: The dialog instance must be declared in local scope if the dialog will be displayed again and again. do
// NOT declare the dialog instance as instance variable of a parent class
/*
    MyDialog dlg; // CORRECT
    dlg.setVisible(thisHwnd);

    class ParentClass {
        MyDialog dlg; // INCORRECT
        ...
    }
*/

/**
 * @class Dialog
 * @brief Represents a modal or modeless dialog window with customizable content.
 *
 * This class manages a dialog window including its title, size, visibility,
 * exit code, and content pane. It provides overridable hooks for dialog lifecycle events.
 */
class Dialog: public Component
{
    WinHandle _hWndParent;       ///< Handle to the parent window.
    WinHandle _hWndDlg;          ///< Handle to the dialog window.
    GridPanel _contentPane; ///< Content pane container inside the dialog.
    WinHandle _hPanel;           ///< Handle to the content pane native window.
    wstring _title;         ///< Dialog window title.
    long _width, _height;   ///< Width and height of the dialog window.

    long _exitCode;         ///< Exit code after the dialog closes (e.g., DIALOG_OK, DIALOG_CANCEL).


protected:

    /**
     * @brief Retrieves window creation options such as title, styles, and class name.
     * @param title Output parameter for window title.
     * @param widownStyles Output parameter for window styles.
     * @param wndClassName Output parameter for window class name.
     * @param isCustomWndProc Output flag indicating custom window procedure usage.
     * @return TRUE if options retrieved successfully; otherwise FALSE.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the native window has been created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new Dialog instance.
     */
    Dialog();

    /**
     * @brief Destructor.
     */
    virtual ~Dialog();

    /**
     * @brief Sets the dialog's window title.
     *
     * @param title The title string to display on the dialog.
     */
    void setTitle(const wstring& title);

    /**
     * @brief Gets the current dialog title.
     *
     * @return The dialog's title string.
     */
    wstring getTitle();

    /**
     * @brief Sets the size of the dialog window.
     *
     * @param width Width of the dialog in pixels.
     * @param height Height of the dialog in pixels.
     */
    void setSize(long width, long height);

    /**
     * @brief Retrieves the current size of the dialog window.
     *
     * @param width Output parameter for dialog width.
     * @param height Output parameter for dialog height.
     */
    void getSize(long& width, long& height);

    /**
     * @brief Shows the dialog and attaches it to a parent window.
     *
     * @param hWndParent Handle to the parent window.
     * @return TRUE if the dialog was successfully made visible.
     */
    BOOL show(Component *parent);

    /**
     * @brief Closes the dialog with a specific exit code.
     *
     * @param exitCode Exit code to set (e.g., DIALOG_OK, DIALOG_CANCEL).
     */
    void close(long exitCode);

    /**
     * @brief Sets the dialog's exit code without closing.
     *
     * @param exitCode Exit code value.
     */
    void setExitCode(long exitCode);

    /**
     * @brief Gets the current exit code of the dialog.
     *
     * @return The exit code (e.g., DIALOG_OK, DIALOG_CANCEL).
     */
    long getExitCode();

    /**
     * @brief Gets a pointer to the content pane GridPanel.
     *
     * @return Pointer to the GridPanel used as content pane.
     */
    GridPanel* getContentPane();

    /**
     * @brief Gets the native window handle of the content pane.
     *
     * @return WinHandle of the content pane.
     */
    WinHandle getContentPaneNativeHandle();

    /**
     * @brief Sets the native handle of the dialog window.
     *
     * @param source Debug string or source identifier.
     * @param hDlg Handle to the native dialog window.
     */
    void setDialogNativeHandle(const wstring& source, WinHandle hDlg);

    /**
     * @brief Called when the dialog window has been moved.
     */
    void onWindowMoved();

public: // Overridables

    /**
     * @brief Called before the dialog window opens.
     *
     * @return TRUE to allow the dialog to open, FALSE to cancel.
     */
    virtual BOOL windowOpening();

    /**
     * @brief Called after the dialog window has opened.
     *
     * @return TRUE if the dialog opened successfully.
     */
    virtual BOOL windowOpened();

    /**
     * @brief Called to determine if the dialog can close.
     *
     * @return TRUE to allow closing, FALSE to prevent.
     */
    virtual BOOL okToClose();

    /**
     * @brief Called after the dialog has closed.
     */
    virtual void windowClosed();
};

