#pragma once

#include "PanelBase.h"
#include "Scrollbar.h"

/**
 * @class FluidPanel
 * @brief A flexible panel layout component that arranges child components fluidly.
 *
 * The FluidPanel manages dynamic layout of child components with scrolling support.
 * It allows setting minimum column width and manages visible rows with scrolling.
 */
class FluidPanel : public PanelBase, public ScrollbarListener
{
    Scrollbar _scrollBar;

    long _margin;                  ///< Sets the left, top, right, and bottom margins
    long _vGap;                    ///< Sets the gap between rows
    long _columnMinWidth;                ///< Minimum width for columns
    long _totalHeightInPixels;     ///< Total height in pixels of the content
    long _rowHeight;               ///< Height of a single row
    long _maxVisibleRows;          ///< Maximum number of visible rows
    long _firstVisibleRow;         ///< Index of the first visible row (scroll offset)
    long _totalRowCount;

    /**
     * @brief Performs the layout of child components.
     *
     * Arranges components based on the minimum column width and available space.
     */
    void Layout();

protected:

    /**
     * @brief Returns the class name for the component.
     * @return Class name string.
     */
    wstring getClassName();

    /**
     * @brief Called when the native window has been created.
     */
    void windowCreated();

public:

    /**
     * @brief Constructor.
     */
    FluidPanel();

    /**
     * @brief Destructor.
     */
    virtual ~FluidPanel();

    /**
     * @brief Clears all child components from the panel.
     */
    void clear();

    /**
     * @brief Sets the height of the field row, excluding the label
     */
    void setFieldHeight(int fieldHeight);

    /**
     * @brief Sets the left, top, right, and bottom margins
     *
     * @param margin margin width/height.
     */
    void setMargin(int margin);

    /**
     * @brief Sets the minimum width for columns.
     * @param minWidth Minimum width in pixels.
     */
    void setMinColumnWidth(long minWidth);

    /**
     * @brief Adds a component with an associated label text to the panel.
     * @param component Pointer to the child component.
     * @param labelText Text label associated with the component.
     */
    void addComponent(Component * component, const wstring& labelText);

    /**
     * @brief Handles window resize events.
     */
    void onWindowResized();

    /**
     * @brief Handles paint events.
     * @param hDC Device context handle for drawing.
     */
    void onPaint(Graphics *g);

    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();

public:

    static int HEIGHT_TEXTFIELD;  ///< Standard height for text fields
    static int HEIGHT_COMBOBOX;   ///< Standard height for combo boxes
    static int HEIGHT_TEXTAREA;   ///< Standard height for text areas
    static int HEIGHT_DATETIME;   ///< Standard height for date/time pickers
    static int HEIGHT_LISTBOX;    ///< Standard height for list boxes
};
