#pragma once

#include "Common.h"
#include "Theme.h"
#include "Models.h"

#define STANDARD_FONT_NAME	L"Arial"
#define STANDARD_FONT_SIZE	12

enum class IconSourceType {
    Invalid = 0,
    File = 1,
    Stock = 2,
    Resource = 3
};

enum class IconType {
    ADD = 1,
    ADDONS = 2,
    ATTACH = 3,
    BANNER = 4,
    BARCODE = 5,
    BINOCULAR = 6,
    BUSINESSPROFILE = 7,
    BUSINESSTYPE = 8,
    CALENDAR = 9,
    CALCULATOR = 10,
    CAMERA = 11,
    CANCEL = 12,
    CASHREGISTER = 13,
    CATEGORY = 14,
    CHECKLIST = 15,
    CHECKMARK = 16,
    CLEAR = 17,
    COLORS = 18,
    COMMISSION = 19,
    CONNECTION = 20,
    CONTRACTTERMS = 21,
    COPY = 22,
    CREDITCARD = 23,
    CUSTOMER = 24,
    DATABASE = 25,
    REMOVE = 26,
    DEPARTMENT = 27,
    DETACH = 28,
    DIAMONDKARAT = 29,
    DISK = 30,
    DOWNLOAD = 31,
    EDIT = 32,
    EMPLOYEE = 33,
    ENVELOPE = 34,
    EXIT = 35,
    EXPENSES = 36,
    EXPORT = 37,
    EYE = 38,
    FILTER = 39,
    FUELDELIVERY = 40,
    FUELPUMP = 41,
    FUELTANK = 42,
    FUELTYPE = 43,
    GIFTCARD = 44,
    HELP = 45,
    HOME = 46,
    IMAGE = 47,
    IMPORT = 48,
    INVENTORY = 49,
    INVOICE = 50,
    JOBTITLES = 51,
    LAYAWAY = 52,
    LINK = 53,
    LOCK = 54,
    LOGIN = 55,
    LOGOUT = 56,
    MARKETING = 57,
    MODIFIERS = 58,
    MONEYBILL = 59,
    NOTE = 60,
    OK = 61,
    PASSWORD = 63,
    PAYMENTGATEWAY = 64,
    PAYMENTMETHODS = 65,
    PAYMENTS = 66,
    PAYROLLENTRIES = 67,
    PAYROLLHOURS = 68,
    PO = 69,
    PRINTER = 70,
    PRODUCT = 71,
    PRODUCTCOST = 72,
    PROMOTION = 73,
    QUOTE = 74,
    RECEIVEDINVENTORY = 75,
    REFRESH = 76,
    REFUNDS = 77,
    RENTAL = 78,
    REPAIR = 79,
    REPORT = 80,
    RETURNS = 81,
    SEARCH = 82,
    SERVICE = 83,
    SETTINGS = 84,
    SHOPPINGBASKET = 85,
    SHOWALL = 86,
    SMS = 87,
    SUPPLIER = 88,
    TAG = 89,
    TAX = 90,
    TODO = 91,
    TRASH = 92,
    UPLOAD = 93,
    USER = 94,
    ALIGNLEFT = 95,
    ALIGNCENTER = 96,
    ALIGNRIGHT = 97,
    ALIGNTOP = 98,
    ALIGNMIDDLE = 99,
    ALIGNBOTTOM = 100,
    RESTOREWINDOW = 101,
    MAXIMIZEWINDOW = 102,
    FOLDEROPENED = 103,
    FOLDERCLOSED = 104,
    DOCUMENT = 105
};

class IconSource
{
    IconSourceType _iconFormat;

public:

    IconSource()
    {
        _iconFormat = IconSourceType::Invalid;
    }

    IconSource(const wstring& iconFile)
    {
        _iconFile = iconFile;
        _iconFormat = iconFile.length() > 0 ? IconSourceType::File : _iconFormat;
    }

    IconSource(const IconType& iconType)
    {
        _iconType = iconType;
        _iconFormat = IconSourceType::Stock;
    }

    IconSource(unsigned int resourceId, const ImageType& imageContentType)
    {
        _resourceId = resourceId;
        _resourceContentType = imageContentType;
        _iconFormat = IconSourceType::Resource;
    }

    BOOL isValid() const
    {
        return _iconFormat != IconSourceType::Invalid;
    }

    IconSourceType getType() const
    {
        return _iconFormat;
    }

    wstring _iconFile;
    IconType _iconType;
    unsigned int _resourceId;
    ImageType _resourceContentType;
};

class Graphics
{
    WinHandle _hWnd;
    DCHandle _hDC;
    HPEN _hPen, _hOldPen;
    HFONT _hFont, _hOldFont;
    HRGN _clipRgn;

public:

    Graphics(WinHandle hWnd, DCHandle hDC);
    virtual ~Graphics();

    DCHandle getHDC();

    COLORREF setTextColor(COLORREF newColor);
    COLORREF getTextColor();
    COLORREF setBackgroundColor(COLORREF newColor);
    COLORREF getBackgroundColor();
    int setBackgroundMode(int bkMode/*TRANSPARENT|OPAQUE|...*/);

    /**
     * @brief Calculates the height of text for a given window.
     * @param defaultHeight Default height if calculation fails.
     * @return Calculated text height.
     */
    int getTextHeight(int defaultHeight);

    /**
     * @brief Calculates the width of the specified text for a given window.
     * @param text Text string to measure.
     * @return Calculated text width in pixels.
     */
    int getTextWidth(const wstring& text);

    void setPen(int width, COLORREF color);
    void restorePen();

    void setFont(const wstring& fontName, int fontSize, BOOL isBold = FALSE, BOOL isItalic = FALSE, BOOL isUnderlined = FALSE);
    void restoreFont();

    /**
     * @brief Draws a line on the device context.
     * @param x1 Starting X coordinate.
     * @param y1 Starting Y coordinate.
     * @param x2 Ending X coordinate.
     * @param y2 Ending Y coordinate.
     * @param color Line color (default is Theme::LineColor).
     */
    void drawLine(long x1, long y1, long x2, long y2, COLORREF color = Theme::LineColor);

    /**
     * @brief Draws a border around the specified rectangle.
     * @param rect Rectangle area.
     * @param color Border color (default is Theme::LineColor).
     */
    void drawBorder(Bounds rect, COLORREF color = Theme::LineColor);

    /**
     * @brief Draws a raised (3D) border around the rectangle.
     * @param rect Rectangle area.
     */
    void drawRaisedBorder(Bounds rect);

    /**
     * @brief Draws a sunken (3D) border around the rectangle.
     * @param rect Rectangle area.
     */
    void drawSunkenBorder(Bounds rect);

    /**
     * @brief Draws a button with the specified text and style.
     * @param rect Rectangle area of the button.
     * @param text Button label text.
     * @param btnType Button style/type (e.g., L"primary"|L"default"|L"danger"|L"success").
     * @param isPressed Button pressed state.
     * @param isDisabled Button disabled state.
     * @param iconFile Icon File Path - Optional (e.g., L"images/filename.png").
     * @param iconPosition Icon orientation (e.g., L"left"|L"top"|L"bottom"|L"right").
     * @param isRounded Button rounded border state.
     * @param hasShadow Button shadow state.
     * @param isGradient Button gradient paint.
     */
    void drawButton(Bounds rect, const wstring& text, const ButtonType& btnType = ButtonType::DEFAULT,                        
                        BOOL isPressed = FALSE, BOOL isDisabled = FALSE,
                        const IconSource& iconSource = IconSource(), const Direction& iconPosition = Direction::LEFT,
                        BOOL isRounded = FALSE, BOOL hasShadow = TRUE, BOOL hasBorder = TRUE,
                        BOOL isGradient = FALSE,
                        COLORREF bkColor = Theme::ButtonBackgroundStartColor, /* Will only be used is btnType is set to ButtonType::CUSTOM */
                        COLORREF fgColor = Theme::ButtonForegroundColor /* Will only be used is btnType is set to ButtonType::CUSTOM */);

    /**
     * @brief Fills a rectangle with a vertical gradient.
     * @param rect Rectangle area to fill.
     * @param bkStartColor Starting color of the gradient.
     * @param bkEndColor Ending color of the gradient.
     */
    void fillGradient(Bounds rect, COLORREF bkStartColor, COLORREF bkEndColor);

    /**
     * @brief Fills a rectangle with the specified color.
     * @param rect Rectangle area to fill.
     * @param color fill color.
     */
    void fillRect(Bounds rect, COLORREF color);

    /**
     * @brief Draws text within the specified rectangle with formatting.
     * @param rect Rectangle area.
     * @param text Text to draw.
     * @param alignment Text alignment (-1 = left, 0 = center, 1 = right).
     * @param bold Whether the text is bold.
     * @param italic Whether the text is italic.
     * @param underlined Whether the text is underlined.
     */
    void drawText(Bounds rect, const wstring& text, int alignment/*-1|0|1*/, 
                         BOOL bold = FALSE, BOOL italic = FALSE, BOOL underlined = FALSE);

    /**
     * @brief Paints a header area with text.
     * @param rect Rectangle area of the header.
     * @param text Header text.
     */
    void paintHeader(Bounds rect, const wstring& text);

    /**
     * @brief Paints a checkbox.
     * @param rect Rectangle area of the checkbox.
     * @param checked The checked state.
     */
    void paintCheckBox(Bounds rect, BOOL checked);

    /**
     * @brief Paints a radio button.
     * @param rect Rectangle area of the radio button.
     * @param checked The checked state.
     */
    void paintRadioButton(Bounds rect, BOOL checked);

    /**
     * @brief Paints a sort direction indicator.
     * @param rect Rectangle area of the sort indicator.
     * @param sortDirection The direction of the sort.
     */
    void paintSortIndicator(Bounds rect, const SortType& sortDirection, COLORREF color);

    void polygon(const POINT* pts, int ptCount, COLORREF fillColor = Theme::BackgroundColor);
    void ellipse(Bounds rect, COLORREF fillColor = Theme::BackgroundColor);
    void rectangle(Bounds rect, COLORREF fillColor = Theme::BackgroundColor);
    void roundRect(Bounds rect, int xRadius, int yRadius, COLORREF bkColor = Theme::BackgroundColor, COLORREF color = Theme::LineColor);

    void setClipRegion(int x1, int y1, int x2, int y2);
    void clearClipRegion();

    void drawDownArrow(Bounds rect, COLORREF color = Theme::ForegroundColor);
    void drawUpArrow(Bounds rect, COLORREF color = Theme::ForegroundColor);
    void drawLeftArrow(Bounds rect, COLORREF color = Theme::ForegroundColor);
    void drawRightArrow(Bounds rect, COLORREF color = Theme::ForegroundColor);

    void drawCheckMark(Bounds rect, COLORREF color = Theme::ForegroundColor);
};
