#pragma once
#include "Common.h"
#include "Graphics.h"

/**
 * @class GraphicsContext
 * @brief Provides static methods for GDI+ initialization and image rendering.
 *
 * This class offers utility functions to initialize and terminate GDI+,
 * load images from files, and draw images onto device contexts with alignment and optional borders.
 */
class GraphicsContext
{

    static IStream* CreateStreamFromMemory(const BYTE* buffer, size_t size);

public:

    /**
     * @brief Initializes the GDI+ library.
     *
     * Must be called before using any GDI+ functionality.
     */
    static void initGdiPlus();

    /**
     * @brief Terminates the GDI+ library and frees resources.
     *
     * Should be called when GDI+ is no longer needed.
     */
    static void terminateGdiPlus();

    /**
     * @brief Loads an image from a file path into a GDI+ Image object.
     * @param filePath The path to the image file.
     * @return Pointer to the loaded Gdiplus::Image object, or nullptr if loading fails.
     */
    static Gdiplus::Image* imageFileToImage(const wstring& filePath);

    /**
     * @brief Loads an image from a byte array into a GDI+ Image object.
     * @return Pointer to the loaded Gdiplus::Image object, or nullptr if loading fails.
     */
    static Gdiplus::Image* imageBytesToImage(const BYTE* imageBytes, size_t size);

    /**
     * @brief Loads a PNG, GIF, or JPG image using its resource ID from .rc file as a GDI+ Image object.
     * @return Pointer to the loaded Gdiplus::Image object, or nullptr if loading fails.
     */
    static Gdiplus::Image* imageResourceToImage(HINSTANCE hInst, int resourceID, const ImageType& imageType);

    /**
     * @brief Draws an image loaded from a file onto a device context.
     * @param hDC Handle to the device context where the image will be drawn.
     * @param rect The target rectangle in which to draw the image.
     * @param filePath The path to the image file.
     * @param alignment Specifies the image alignment within the rectangle. Default is "topCenter".
     * @param showBorder If TRUE, draws a border around the image. Default is FALSE.
     */
    static void drawImageFile(Graphics *g, Bounds rect, const wstring& filePath,
                              const ImageAlignment& alignment = ImageAlignment::TOPCENTER, BOOL showBorder = FALSE);

    /**
     * @brief Draws a GDI+ Image onto a device context.
     * @param hDC Handle to the device context where the image will be drawn.
     * @param rect The target rectangle in which to draw the image.
     * @param image Pointer to the Gdiplus::Image to draw.
     * @param alignment Specifies the image alignment within the rectangle. Default is "topCenter".
     * @param showBorder If TRUE, draws a border around the image. Default is FALSE.
     */
    static void drawImage(Graphics *g, Bounds rect, Gdiplus::Image* image,
                        const ImageAlignment& alignment = ImageAlignment::TOPCENTER, BOOL showBorder = FALSE);

    /**
     * @brief Draws an image represented by a byte array onto a device context.
     * @param hDC Handle to the device context where the image will be drawn.
     * @param rect The target rectangle in which to draw the image.
     * @param imageBytes Pointer to the image bytes.
     * @param size the size of the imageBytes array.
     * @param alignment Specifies the image alignment within the rectangle. Default is "topCenter".
     * @param showBorder If TRUE, draws a border around the image. Default is FALSE.
     */
    static void drawImageBytes(Graphics *g, Bounds rect, const BYTE* imageBytes, size_t size,
        const ImageAlignment& alignment = ImageAlignment::TOPCENTER, BOOL showBorder = FALSE);

    /**
     * @brief Draws a GDI+ Image onto a device context.
     * @param hDC Handle to the device context where the image will be drawn.
     * @param rect The target rectangle in which to draw the image.
     * @param resourceID ID from the .rc resource
     * @param alignment Specifies the image alignment within the rectangle. Default is "topCenter".
     * @param showBorder If TRUE, draws a border around the image. Default is FALSE.
     */
    static void drawImageResource(Graphics *g, Bounds rect, int resourceID, const ImageType& imageType,
        const ImageAlignment& alignment = ImageAlignment::TOPCENTER, BOOL showBorder = FALSE);

    /**
     * @brief Fills a rect with a linear gradient colors.
     */
    static void linearGradient(Graphics *g, Bounds rect, int startR, int startG, int startB, 
                                int endR, int endG, int endB, 
                                int startAlpha = 0, int endAlpha = 0, BOOL directionVertical = TRUE,
                                const wstring& text = L"", int textR = 0, int textG = 0, int textB = 0);
};

