#pragma once

#include "Component.h"
#include "TextField.h"
#include "MaskedField.h"
#include "ComboBox.h"
#include "Models.h"
#include "DateTimePicker.h"
#include "CheckBox.h"
#include "Scrollbar.h"

class GridColumn
{
public:
    GridColumn()
    {
        _isVisible = TRUE;
        _isEditable = FALSE;
        _editor = NULL;
        _editStyle = EditType::TEXTFIELD;
        _editOptions = {};

        _columnStartX = 0;
        _columnEndX = 0;
        _sortIndicatorRect = { 0,0,0,0 };
    }

    wstring _title, _name;   
    long _alignment;                
    long _widthRatio;
    long _calculatedWidth;
    int _columnStartX, _columnEndX;
    Bounds _sortIndicatorRect;
    
    BOOL _isVisible;
    BOOL _isSortable;
    BOOL _isFilterable;
    BOOL _isEditable;
    wstring _computedColumnExpression;
    EditType _editStyle;
    vector<wstring> _editOptions;
    Component* _editor;
};

class GridRow
{
public:
    vector<wstring> _values; 
    Bounds _rect;              
};

/**
 * @class Grid
 * @brief A UI component representing a data grid with paging, selection, and editing capabilities.
 * 
 * Supports paging via an associated Pager object, configurable columns with editors,
 * scrolling, and user interaction through keyboard and mouse.
 */
class Grid : public Component, public ScrollbarListener
{
    Scrollbar _scrollBar;

    GridChangeListener* _gridChangeListener; ///< Listener to cell value changes

    Pager* _pager;                ///< Pager for managing data paging
    PagerRequest _pagerRequest;   ///< Pager request info
    BOOL _pageFetched;            ///< Indicates if a page has been fetched
    BOOL _isLoadingPage;          ///< Indicates if a page load is in progress
    __int64 _lastLoadedFirstVisibleRow;
    __int64 _lastLoadedMaxVisibleRows;
    __int64 _totalRowCount;       ///< Total number of rows available

    wstring _sortedBy;            ///< Indicates the name of the column the data is currently sorted by
    SortType _sortDirection;     ///< Indicates the direction of the current sort order

    Bounds _pageHomeBtnRect;        ///< Rectangle for the "first page" button
    Bounds _pagePrevBtnRect;        ///< Rectangle for the "previous page" button
    Bounds _pageNextBtnRect;        ///< Rectangle for the "next page" button
    Bounds _pageEndBtnRect;         ///< Rectangle for the "last page" button

    vector<GridColumn> _columns;  ///< List of columns in the grid
    vector<GridRow> _rows;        ///< List of rows in the current page

    long _maxVisibleRows;         ///< Maximum rows visible in the current view
    __int64 _firstVisibleRow;        ///< Index of the first visible row
    __int64 _lastVisibleRow;         ///< Index of the last visible row
    __int64 _currentRow;             ///< Index of the currently selected row
    long _currentColumn;          ///< Index of the currently selected cell column#

    BOOL _isStriped;            ///< If true, the row colors alternate

    TextField _textEditor;        ///< Text Cell Editor
    MaskedField _maskEditor;        ///< Mask Cell Editor
    ComboBox _comboEditor;        ///< Combo Cell Editor
    Component* _currentCellEditor;  ///< a pointer to the current active type of editor (i.e. _textEditor, _maskEditor, ...)
    EditType _currentCellEditorStyle;
    __int64 _editorRowIndex, _editorColumnIndex;

    void FixRow(__int64 rowIndex);   ///< Ensures row layout and data consistency
    void LayoutColumns();         ///< Calculates and arranges column widths and positions

    void ResetPaging();           ///< Resets paging state and reloads first page
    void LoadPage(BOOL doPaint = TRUE); ///< Loads data for the current page, optionally repaints

    void SetCurrentCell(__int64 rowIndex, long columnIndex, BOOL& isNewCellEditable);
    void HideEditor();

    BOOL FireCellValueChangeEvent(__int64 rowIndex, const wstring& columnName, const wstring& oldValue, const wstring& newValue);

protected:
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc) override;
    void windowCreated() override;

public:
    Grid();
    virtual ~Grid();

    void setPager(Pager* pager); ///< Assigns a pager for handling data loading
    void reloadPage();
    void resetPaging();

    void setStriped(BOOL isStriped);

    void clear();                ///< Clears all rows and resets the grid
    void clearRows();

    vector<GridColumn> getColumns(); ///< Returns all grid columns
    void addColumn(const wstring& name, const wstring& title, BOOL isVisible, UINT widthRatio, long alignment = -1, 
        BOOL isSortable = FALSE, BOOL isFilterable = FALSE, BOOL isEditable = FALSE,
        const EditType& editStyle = EditType::TEXTFIELD, 
        const vector<wstring>& editOptions = {},
        const wstring& computedColumnExpression = L""); ///< Adds a new column

    void setColumnEditor(long columnIndex, Component* component); ///< Sets an editor component for a column
    Component* getColumnEditor(long columnIndex);                 ///< Gets the editor component for a column

    __int64 getRowCount();          ///< Returns the number of rows currently loaded
    __int64 getSelectedRow();       ///< Returns the index of the currently selected row
    __int64 addRow();               ///< Adds a new row and returns its index

    void setCellValue(__int64 rowIndex, long columnIndex, const wstring& value); ///< Sets the value of a specific cell
    wstring getCellValue(__int64 rowIndex, long columnIndex);                   ///< Gets the value of a specific cell
    void setCellValue(__int64 rowIndex, const wstring& columnName, const wstring& value); ///< Sets the value of a specific cell
    wstring getCellValue(__int64 rowIndex, const wstring& columnName);                   ///< Gets the value of a specific cell

    BOOL deleteRow(__int64 rowIndex); ///< Deletes a row at the given index

public: // Overridables

    void onPaint(Graphics *g) override;
    void onWindowResized() override;

    void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onPageUp(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onPageDown(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onKeyTab(BOOL shiftPressed, BOOL ctrlPressed) override;

    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed) override;
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta) override;

    void onFocusGained() override;
    void onFocusLost() override;

    void addSelectionChangedListener(SelectionChangeListener* l);
    void addItemDoubleClickedListener(RowDoubleClickListener* l);
    
    void addGridChangeListener(GridChangeListener* l);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
