#pragma once

#include "Component.h"
#include "Scrollbar.h"

class IconMenuItem
{
public:
    BOOL _imageIsResourceID;
    int _imageResourceID;
    ImageType _imageType;
    wstring _imageFile;
    wstring _text;       
    Bounds _rect;          
    Bounds _imageRect;     
    Bounds _textRect;      
};

/**
 * @class IconMenu
 * @brief A menu component that displays a list of items, each with an icon and text.
 * 
 * Supports scrolling, keyboard navigation, and mouse interaction.
 */
class IconMenu : public Component, public ScrollbarListener
{
    Scrollbar _scrollBar;

    wstring _title;                         ///< Title displayed above the menu.
    vector<IconMenuItem> _items;            ///< List of menu items.

    long _maxVisibleItems;                  ///< Maximum number of visible items in the viewport.
    __int64 _firstVisibleItem;                 ///< Index of the first visible item.
    __int64 _lastVisibleItem;                  ///< Index of the last visible item.
    __int64 _currentItem;                      ///< Currently selected item index.

    __int64 _indexOfPressedItem;            /**< Index of the currently pressed item. */

    BOOL _isPinnable;

protected:
    /**
     * @brief Sets window creation options specific to the IconMenu.
     * @param title Output parameter to receive window title.
     * @param widownStyles Output parameter to receive window styles.
     * @param wndClassName Output parameter to receive window class name.
     * @param isCustomWndProc Output parameter indicating if a custom window procedure is used.
     * @return TRUE if options are successfully set.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called once the native window is created; performs initialization.
     */
    void windowCreated();

public:
    IconMenu();
    virtual ~IconMenu();

    /**
     * @brief Removes all menu items and clears the menu.
     */
    void clear();

    /**
     * @brief Sets the title displayed at the top of the menu.
     * @param title The title string.
     */
    void setTitle(const wstring& title);

    void setPinnable(BOOL isPinnable);

    /**
     * @brief Adds a new item to the menu.
     * @param imageFile Path to the icon image file for the item.
     * @param itemText Text label for the item.
     * @return The index of the newly added item.
     */
    long addItem(const wstring& imageFile, const wstring& itemText);
    long addItem(int imageResourceID, const ImageType& imageType, const wstring& itemText);

    /**
     * @brief Paints the menu, drawing visible items with their icons and text.
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Handles resizing of the menu window, recalculating layout.
     */
    void onWindowResized();

    /**
     * @brief Handles keyboard navigation: moves selection up.
     */
    void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles keyboard navigation: moves selection down.
     */
    void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles page-up keyboard navigation.
     */
    void onPageUp(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles page-down keyboard navigation.
     */
    void onPageDown(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles keyboard Home key navigation: moves selection to first item.
     */
    void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles keyboard End key navigation: moves selection to last item.
     */
    void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles mouse press events, selecting or interacting with items.
     */
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles mouse right-click events, possibly for context menu or other actions.
     */
    void onMouseRightClicked(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles mouse wheel scrolling for scrolling through menu items.
     * @return TRUE if the event was handled, FALSE otherwise.
     */
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);

    /**
     * @brief Adds a listener to receive selection change events.
     * @param l Pointer to a SelectionChangeListener.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
