#pragma once

#include "Component.h"

/**
 * @class Image
 * @brief A GUI component that displays an image with configurable alignment.
 * 
 * Supports setting the image file and alignment such as topLeft, middleCenter, bottomRight, etc.
 */
class Image : public Component
{
    BOOL _imageIsResourceID;   ///< Indicates whether the resourceID or image file is specified.
    int _imageResourceID;   ///< Resource id of the image.
    ImageType _imageType;   ///< Required for images specified as resource id
    wstring _imageFile;   ///< Path to the image file to display.
    ImageAlignment _alignment;   ///< Alignment of the image within the component. Valid values:
                          ///< topLeft, topCenter, topRight,
                          ///< middleLeft, middleCenter, middleRight,
                          ///< bottomLeft, bottomCenter, bottomRight.

protected:
    /**
     * @brief Provides options required to create the native window for the image component.
     * 
     * @param title [out] The window title.
     * @param widownStyles [out] The window styles.
     * @param wndClassName [out] The window class name.
     * @param isCustomWndProc [out] Whether a custom window procedure is used.
     * @return TRUE if options set successfully.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called when the native window has been created; used for initialization.
     */
    void windowCreated();

public:
    Image();
    virtual ~Image();

    /**
     * @brief Sets the image file to display.
     * 
     * @param imageFile Path to the image file.
     */
    void setImage(const wstring& imageFile);
    void setImage(int resourceID, const ImageType& imageType);

    /**
     * @brief Sets the alignment of the image within the component.
     * 
     * @param alignment Alignment string. Possible values:
     * "topLeft", "topCenter", "topRight",
     * "middleLeft", "middleCenter", "middleRight",
     * "bottomLeft", "bottomCenter", "bottomRight".
     */
    void setAlignment(const ImageAlignment& alignment);

    /**
     * @brief Paints the image inside the component, respecting the alignment.
     * 
     * @param hDC Handle to the device context to paint on.
     */
    void onPaint(Graphics *g);
};
