#pragma once

#include "GraphicsContext.h"

/**
 * @class ImageCache
 * @brief Manages caching of GDI+ images to improve rendering performance.
 * 
 * This class stores images keyed by a string, loads images from files,
 * and draws cached images to device contexts with specified alignment and optional border.
 */
class ImageCache
{
    map<wstring, Gdiplus::Image*> _imageCache; ///< Map of image keys to Gdiplus::Image pointers.

public:

    /**
     * @brief Constructs an empty ImageCache.
     */
    ImageCache();

    /**
     * @brief Destructor clears and releases all cached images.
     */
    virtual ~ImageCache();

    /**
     * @brief Clears all cached images, releasing their memory.
     */
    void clear();

    /**
     * @brief Loads an image from a file and adds it to the cache with the specified key.
     * 
     * @param key The string key to identify the cached image.
     * @param filePath The file path of the image to load.
     * @return TRUE if the image was loaded and cached successfully, FALSE otherwise.
     */
    BOOL addImage(const wstring& key, const wstring& filePath);

    /**
     * @brief Draws a cached image identified by key onto a device context.
     * 
     * @param hDC Handle to the device context where the image will be drawn.
     * @param key The key identifying the cached image.
     * @param rect The target rectangle for drawing the image.
     * @param alignment Alignment of the image within the rectangle (e.g., "topLeft", "middleCenter").
     * @param showBorder If TRUE, draws a border around the image.
     * @return TRUE if the image was found and drawn successfully, FALSE otherwise.
     */
    BOOL drawImage(Graphics *g, const wstring& key, Bounds rect, const ImageAlignment& alignment, BOOL showBorder);
};
