#pragma once

#include "Component.h"

/**
 * @class MaskedField
 * @brief Input field that supports input masking and numeric input.
 * 
 * Supports masks where:
 * - '#' represents a digit
 * - '@' represents an alpha character
 * - '*' represents an alphanumeric character
 */
class MaskedField : public Component
{
    wstring _mask;                ///< Input mask pattern
    wstring _text;                ///< Current text value
    BOOL _isNumeric;              ///< Flag if the field is numeric
    long _numericDecimalDigits;   ///< Number of decimal digits allowed if numeric
    BOOL _readOnly;

protected:

    /**
     * @brief Provides window creation options for the masked input field.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the window is created.
     */
    void windowCreated();

public:

    /**
     * @brief Constructs a MaskedField with no mask or text.
     */
    MaskedField();

    /**
     * @brief Destructor.
     */
    virtual ~MaskedField();

    /**
     * @brief Returns the preferred height
     */
    int getPreferredHeight();

    void setReadOnly(BOOL readOnly);
    BOOL isReadOnly();

    /**
     * @brief Configures the field to accept numeric input only, with specified decimal digits.
     * 
     * @param decimalDigits Number of digits allowed after the decimal point.
     */
    void setNumeric(long decimalDigits);

    /**
     * @brief Checks if the field is numeric and outputs decimal digits count.
     * 
     * @param decimalDigits Output parameter for decimal digits.
     * @return TRUE if numeric; FALSE otherwise.
     */
    BOOL isNumeric(long& decimalDigits);

    /**
     * @brief Sets the input mask for the field.
     * 
     * Valid characters in mask:
     *  - '#' for digits
     *  - '@' for alphabetic characters
     *  - '*' for alphanumeric characters
     * 
     * @param mask Mask string pattern.
     */
    void setMask(const wstring& mask);

    /**
     * @brief Retrieves the current mask.
     * 
     * @return Mask string.
     */
    wstring getMask();

    /**
     * @brief Sets the text value of the field.
     * 
     * @param text Input text.
     */
    void setText(const wstring& text);

    /**
     * @brief Gets the current text value.
     * 
     * @return Current text string.
     */
    wstring getText();

    /**
     * @brief Handles data changed events.
     *
     * @param hTarget Handle to the target window.
     * @param newValue The new text value.
     * @param procParams Additional parameters.
     */
    void onDataChanged(WinHandle hTarget, const wstring& newValue, const ProcParams& procParams);

    /**
     * @brief Adds a listener to receive data change notifications.
     * 
     * @param l Pointer to a DataChangeListener.
     */
    void addDataChangedListener(DataChangeListener* l);
};
