#pragma once

#include "Component.h"

/**
 * @brief A star rating UI component that supports whole and half stars.
 */
class Rating : public Component
{
    int _savedRating = 2;      /**< Currently saved rating value (number of stars) */
    BOOL _hasHalf = TRUE;      /**< Whether half-star ratings are enabled */
    Bounds _starRects[5];        /**< Rectangles defining each star's position and size */
    int _hoverRating = _savedRating; /**< Rating value shown on mouse hover */
    BOOL _isHovering = FALSE;  /**< True if mouse is currently hovering over the rating */

    /**
     * @brief Draws a full star in the specified rectangle.
     * 
     * @param hDC Device context handle for drawing.
     * @param rect Rectangle area to draw the star.
     * @param selected If true, draws the star as selected (filled), otherwise unselected.
     */
    void DrawStar(Graphics *g, Bounds rect, BOOL selected = FALSE);

    /**
     * @brief Draws a half star in the specified rectangle.
     * 
     * @param hDC Device context handle for drawing.
     * @param rect Rectangle area to draw the half star.
     * @param selected If true, draws the half star as selected (filled), otherwise unselected.
     */
    void DrawHalfStar(Graphics* g, Bounds rect, BOOL selected = FALSE);

protected:
    /**
     * @brief Gets options for creating the window.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called when the window is created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new Rating component.
     */
    Rating();

    /**
     * @brief Destructor for the Rating component.
     */
    virtual ~Rating();

    /**
     * @brief Returns the preferred height
     */
    int getPreferredHeight();

    /**
     * @brief Sets the current rating and whether half-star ratings are allowed.
     * 
     * @param rating Number of stars (can include halves if enabled).
     * @param includesHalf TRUE if half-star ratings are enabled.
     */
    void setRating(int rating, BOOL includesHalf);

    /**
     * @brief Gets the current rating and half-star status.
     * 
     * @param rating Output parameter for the current rating.
     * @param includesHalf Output parameter indicating if half-star ratings are enabled.
     */
    void getRating(int& rating, BOOL& includesHalf);

    /**
     * @brief Paints the rating control.
     * 
     * @param hDC Device context handle for painting.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Handles mouse press events on the control.
     * 
     * @param hWnd Handle to the window.
     * @param x X coordinate of the mouse click.
     * @param y Y coordinate of the mouse click.
     * @param clickCount Number of clicks.
     * @param shiftPressed TRUE if Shift key was pressed.
     * @param ctrlPressed TRUE if Ctrl key was pressed.
     */
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles mouse move events over the control.
     * 
     * @param hWnd Handle to the window.
     * @param x X coordinate of the mouse.
     * @param y Y coordinate of the mouse.
     * @param shiftPressed TRUE if Shift key is pressed.
     * @param ctrlPressed TRUE if Ctrl key is pressed.
     */
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
};
