#pragma once

#include "Component.h"

/**
 * @brief A toggle switch UI component with two options.
 */
class Switch : public Component
{
    wstring _option1;          /**< First option text */
    wstring _option2;          /**< Second option text */
    wstring _selectedOption;   /**< Currently selected option */

protected:
    /**
     * @brief Gets options for creating the window.
     * 
     * @param title Reference to window title.
     * @param widownStyles Reference to window styles.
     * @param wndClassName Reference to window class name.
     * @param isCustomWndProc Reference indicating if a custom window procedure is used.
     * @return TRUE if successful, FALSE otherwise.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called when the window is created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new Switch component.
     */
    Switch();

    /**
     * @brief Destructor for the Switch component.
     */
    virtual ~Switch();

    /**
     * @brief Returns the preferred height
     */
    int getPreferredHeight();

    /**
     * @brief Sets the two toggle options.
     * 
     * @param option1 Text for the first option.
     * @param option2 Text for the second option.
     */
    void setOptions(const wstring& option1, const wstring& option2);

    /**
     * @brief Sets the currently selected option.
     * 
     * @param selectedOption The option to select.
     */
    void setSelectedtem(const wstring& selectedOption);

    /**
     * @brief Gets the currently selected option.
     * 
     * @return Selected option text.
     */
    wstring getSelectedtem();

public: // Overridables

    /**
     * @brief Paint handler for the switch control.
     * 
     * @param hDC Device context handle.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Handler called when the window is resized.
     */
    void onWindowResized();

    /**
     * @brief Handler for mouse press events.
     * 
     * @param hWnd Handle to the window.
     * @param x X coordinate of the mouse.
     * @param y Y coordinate of the mouse.
     * @param clickCount Number of clicks.
     * @param shiftPressed TRUE if shift key pressed.
     * @param ctrlPressed TRUE if ctrl key pressed.
     */
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Adds a listener to be notified when the selection changes.
     * 
     * @param l Pointer to a SelectionChangeListener.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);
};
