#pragma once

#include "Component.h"

/**
 * @brief A text input field component.
 */
class TextArea : public Component
{
    wstring _text;      /**< The current text content of the field */
    int _maxLength;
    BOOL _readOnly;

protected:
    /**
     * @brief Retrieves window creation options.
     * 
     * @param title Reference to the window title.
     * @param widownStyles Reference to window styles.
     * @param wndClassName Reference to window class name.
     * @param isCustomWndProc Reference to custom window procedure flag.
     * @return TRUE if successful, FALSE otherwise.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called when the window is created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new TextArea.
     */
    TextArea();

    /**
     * @brief Destructor.
     */
    virtual ~TextArea();

    void setReadOnly(BOOL readOnly);

    void setMaxLength(int maxLen);

    /**
     * @brief Sets the text content of the field.
     * 
     * @param text The text to set.
     */
    void setText(const wstring& text);

    /**
     * @brief Gets the current text content.
     * 
     * @return The current text in the field.
     */
    wstring getText();

    /**
     * @brief Gets the text as separate lines.
     */
    void getLines(vector<wstring>& lines);

    /**
     * @brief Gets the selected text content.
     */
    wstring getSelectedText();
    void replaceSelectedText(const wstring& newText);

    void selectText(int startPos, int endPos);

    BOOL canUndo();
    BOOL undo();

    void setCaretPos(int newPos);

    /**
     * @brief Gets the max allowed length of the text.
     */
    int getTextLimit();

    /**
     * @brief Sets the max allowed length of the text.
     */
    void setTextLimit(int textLimit);

    /**
     * @brief Gets the preferred height of the text field control.
     * 
     * @return Preferred height in pixels.
     */
    int getPreferredHeight();

    /**
     * @brief Handles data changed events.
     * 
     * @param hTarget Handle to the target window.
     * @param newValue The new text value.
     * @param procParams Additional parameters.
     */
    void onDataChanged(WinHandle hTarget, const wstring& newValue, const ProcParams& procParams);

    /**
     * @brief Adds a listener for data changed events.
     * 
     * @param l Pointer to the listener.
     */
    void addDataChangedListener(DataChangeListener* l);
};
