#pragma once

#include "Component.h"

/**
 * @brief A UI component representing a toggle button with multiple selectable options.
 * 
 * Displays a set of toggle options where one can be selected at a time.
 */
class ToggleButton : public Component
{
    vector<Bounds> _optionsRect;          /**< Rectangles defining clickable areas for each option */
    vector<wstring> _options;           /**< List of option strings */
    wstring _selectedButtonText;        /**< Currently selected option text */

protected:
    /**
     * @brief Prepares window creation options.
     * 
     * @param title Reference to window title string to set.
     * @param widownStyles Reference to window styles to set.
     * @param wndClassName Reference to window class name string to set.
     * @param isCustomWndProc Reference flag whether custom window proc is used.
     * @return TRUE if options set successfully, otherwise FALSE.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called when the window has been created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new ToggleButton object.
     */
    ToggleButton();

    /**
     * @brief Destructor.
     */
    virtual ~ToggleButton();

    /**
     * @brief Returns the preferred height
     */
    int getPreferredHeight();

    /**
     * @brief Sets the toggle options.
     * 
     * @param options Vector of option strings.
     */
    void setOptions(const vector<wstring>& options);

    /**
     * @brief Sets the currently selected option by its text.
     * 
     * @param optionText The text of the option to select.
     */
    void setSelectedItem(const wstring& optionText);

    /**
     * @brief Paints the toggle button and its options.
     * 
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Handles mouse press events.
     * 
     * @param hWnd Handle to the window.
     * @param x X-coordinate of mouse.
     * @param y Y-coordinate of mouse.
     * @param clickCount Number of clicks.
     * @param shiftPressed TRUE if Shift key pressed.
     * @param ctrlPressed TRUE if Ctrl key pressed.
     */
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Adds a listener to be notified when the selection changes.
     * 
     * @param l Pointer to a SelectionChangeListener.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);
};
