#pragma once

#include "Component.h"
#include "PopupMenu.h"
#include "Scrollbar.h"

class TreeViewItem
{
public:
    wstring _parentId;           
    wstring _id;                 
    wstring _text;               
    wstring _nodeType;           
    Bounds _expandCollapseRect;    
    Bounds _iconRect;              
    Bounds _textRect;              
    long _indentLevel;           
    BOOL _hasChildren;           
    BOOL _isExpanded;            
};

class TreeViewMenuItem : public PopupMenuItem
{
public:
    wstring _nodeType;          
};

/**
 * @brief TreeView component for displaying hierarchical data with expandable nodes.
 */
class TreeView : public Component, public ScrollbarListener
{
    Scrollbar _scrollBar;

    wstring _title;                              /**< Title of the TreeView */
    vector<TreeViewItem> _allItems;             /**< All items in the tree (including hidden) */
    vector<TreeViewItem> _visibleItems;         /**< Items currently visible (expanded) */

    vector<TreeViewMenuItem> _contextMenuItems; /**< Context menu items for the TreeView */

    long _maxVisibleItems;       /**< Maximum number of visible items in the view */
    __int64 _firstVisibleItem;      /**< Index of the first visible item */
    __int64 _lastVisibleItem;       /**< Index of the last visible item */
    __int64 _currentItem;           /**< Index of the currently selected item */

    /**
     * @brief Recalculates the list of visible items based on expanded/collapsed nodes.
     */
    void ReloadVisibleItems();

protected:
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);
    void windowCreated();

public:
    /**
     * @brief Constructs a new TreeView object.
     */
    TreeView();

    /**
     * @brief Destructor.
     */
    virtual ~TreeView();

    /**
     * @brief Clears all items from the TreeView.
     */
    void clear();

    /**
     * @brief Sets the title of the TreeView.
     * @param title The new title text.
     */
    void setTitle(const wstring& title);

    /**
     * @brief Adds a new item to the TreeView.
     * @param itemText The text for the new item.
     * @param itemId The unique ID for the new item (optional).
     * @param parentId The ID of the parent item (optional).
     * @param nodeType The node type for the item (optional).
     * @return The index of the newly added item.
     */
    long addItem(const wstring& itemText, const wstring& itemId = L"", const wstring& parentId = L"", const wstring& nodeType = L"");
    
    BOOL updateItem(const wstring& itemId, const wstring& itemText);
    
    BOOL deleteItem(const wstring& itemId);

    long addPath(const wstring& itemPath, const wstring& parentId = L"");

    /**
     * @brief Gets the total count of items in the TreeView.
     * @return Number of items.
     */
    long getItemCount();

    BOOL getItem(long itemIndex, wstring& itemText, wstring& itemId, wstring& parentId, wstring& itemPath);

    /**
     * @brief Gets the index of the currently selected item.
     * @return Selected item index.
     */
    long getSelectedItem(wstring& itemText, wstring& itemId, wstring& parentId, wstring& itemPath);

    /**
     * @brief Adds an item to the context menu.
     * @param actionId Action identifier for the menu item.
     * @param text Display text of the menu item.
     * @param nodeType Node type this menu item applies to (optional).
     */
    void addContextMenuItem(long actionId, const wstring& text, const wstring& nodeType = L"");

    /** @name Event handlers */
    ///@{
    void onPaint(Graphics *g);
    void onWindowResized();

    void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed);
    void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed);
    void onPageUp(BOOL shiftPressed, BOOL ctrlPressed);
    void onPageDown(BOOL shiftPressed, BOOL ctrlPressed);
    void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed);
    void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed);

    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseRightClicked(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);
    ///@}

    /**
     * @brief Adds an action listener to respond to user interactions.
     * @param l Pointer to the ActionListener.
     */
    void addActionListener(ActionListener* l);

    /**
     * @brief Adds a listener for selection changes.
     * @param l Pointer to the SelectionChangeListener.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
