#pragma once

#include "Component.h"

class DayInfo
{
public:

	DayInfo(int dayNo, Bounds rect)
	{
		_dayNumber = dayNo;
		_rect.left = rect.left;
		_rect.top = rect.top;
		_rect.right = rect.right;
		_rect.bottom = rect.bottom;
	}
	int _dayNumber;
	Bounds _rect;
};

/**
 * @class Calendar
 * @brief A UI component for displaying and interacting with calendar dates.
 *
 * Inherits from Component and provides functionality to display months,
 * handle user input, and manage selected dates.
 */
class Calendar : public Component
{
	/**
	 * @brief Whether the calendar is shown as a popup.
	 */
	BOOL _isPopup;

	/**
	 * @brief Display orientation of the calendar (e.g., vertical or horizontal).
	 */
	wstring _orientation;

	/**
	 * @brief Rectangles defining the screen area for each of the 12 months.
	 */
	Bounds _monthRects[12];

	/**
	 * @brief A list of rectangles and associated data representing individual days.
	 */
	vector<DayInfo> _dayRects;

	/**
	 * @brief Rectangle representing the area where the current year is displayed.
	 */
	Bounds _yearRect;

	/**
	 * @brief Rectangle to increment the year.
	 */
	Bounds _incYearRect;

	/**
	 * @brief Rectangle to decrement the year.
	 */
	Bounds _decYearRect;

	/**
	 * @brief Rectangle representing today's date.
	 */
	Bounds _todayRect;

	/**
	 * @brief Rectangle representing the selected year number.
	 */
	Bounds _yearNumberRect;

	/**
	 * @brief Indicates if the year is currently being edited.
	 */
	BOOL _isEditingYear;

	/**
	 * @brief Currently displayed year, month, and day.
	 */
	int _curYear, _curMonth;

	/**
	 * @brief The selected year, month, and day.
	 */
	int _selectedYear, _selectedMonth, _selectedDay;

	/**
	 * @brief Called when the selected date changes.
	 */
	void dateChanged();

	/**
	 * @brief Updated _curYear and _curMonth from _selectedYear and _selectedMonth
	 */
	void syncCurrentFromSelected();

protected:

	/**
	 * @brief Retrieves options needed to create the calendar window.
	 *
	 * @param title The window title.
	 * @param widownStyles The style flags for the window.
	 * @param wndClassName The class name of the window.
	 * @param isCustomWndProc Whether a custom window procedure is used.
	 * @return TRUE if successful, FALSE otherwise.
	 */
	BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

	/**
	 * @brief Called after the window has been successfully created.
	 */
	void windowCreated();

public:

	/**
	 * @brief Constructs a Calendar component.
	 */
	Calendar();

	/**
	 * @brief Destroys the Calendar component.
	 */
	virtual ~Calendar();

	/**
	 * @brief Sets whether the calendar is used in a popup context.
	 *
	 * @param isPopup TRUE to mark the calendar as a popup, FALSE otherwise.
	 */
	void setAsPopup(BOOL isPopup);

	/**
	 * @brief Return TRUE if this component is displayed as a popup (e.g. DateTimePicker).
	 */
	BOOL isPopup();

	/**
	 * @brief Sets the orientation of the calendar.
	 *
	 * @param orientation A string indicating the orientation ("vertical" or "horizontal").
	 */
	void setOrientation(const wstring& orientation);

	/**
	 * @brief Sets the selected date using a string formatted as YYYYMMDD.
	 *
	 * @param dateYYYYMMDD Date string in the format "YYYYMMDD".
	 */
	void setSelectedDate(const wstring& dateYYYYMMDD);

	/**
	 * @brief Sets the selected date using year, month, and day integers.
	 *
	 * @param year The year to select.
	 * @param month The month to select.
	 * @param day The day to select.
	 */
	void setSelectedDate(int year, int month, int day);

	/**
	 * @brief Gets the selected date as a string formatted as YYYYMMDD.
	 *
	 * @return The currently selected date string.
	 */
	wstring getSelectedDate();

	/**
	 * @brief Gets the selected date components.
	 *
	 * @param year Output parameter for the selected year.
	 * @param month Output parameter for the selected month.
	 * @param day Output parameter for the selected day.
	 * @return The currently selected date as a string in the format "YYYYMMDD".
	 */
	wstring getSelectedDate(int& year, int& month, int& day);

public: // Overridables

	/**
	 * @brief Paints the calendar to the provided device context.
	 *
	 * @param hDC Handle to the device context.
	 */
	void onPaint(Graphics *g);

	/**
	 * @brief Called when the window has been resized.
	 */
	void onWindowResized();

	/**
	 * @brief Called when the calendar gains focus.
	 */
	void onFocusGained();

	/**
	 * @brief Called when the calendar loses focus.
	 */
	void onFocusLost();
	
	/**
	 * @brief Handles Backspace key input.
	 *
	 * @param shiftPressed Whether the Shift key is pressed.
	 * @param ctrlPressed Whether the Ctrl key is pressed.
	 */
	void onKeyBackSpace(BOOL shiftPressed, BOOL ctrlPressed);

	/**
	 * @brief Handles typed character input.
	 *
	 * @param ch The character typed.
	 * @param shiftPressed Whether the Shift key is pressed.
	 * @param ctrlPressed Whether the Ctrl key is pressed.
	 */
	void onKeyTyped(wchar_t ch, BOOL shiftPressed, BOOL ctrlPressed);

	/**
	 * @brief Handles mouse press events.
	 *
	 * @param hWnd Handle to the window receiving the event.
	 * @param x The x-coordinate of the mouse event.
	 * @param y The y-coordinate of the mouse event.
	 * @param clickCount Number of mouse clicks.
	 * @param shiftPressed Whether the Shift key is pressed.
	 * @param ctrlPressed Whether the Ctrl key is pressed.
	 */
	void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);

	/**
	 * @brief Handles mouse wheel events.
	 *
	 * @param hWnd Handle to the window receiving the event.
	 * @param x The x-coordinate of the mouse event.
	 * @param y The y-coordinate of the mouse event.
	 * @param delta The scroll delta.
	 * @return TRUE if the event was handled, FALSE otherwise.
	 */
	BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);

	/**
	 * @brief Adds a listener for selection change events.
	 *
	 * @param l Pointer to a SelectionChangeListener to notify on selection change.
	 */
	void addSelectionChangedListener(SelectionChangeListener* l);
};
