#pragma once

#include "Component.h"

class Card
{
public:
	Card(const wstring& cardName, Component* component)
	{
		_cardName = cardName;
		_component = component;
	}

	wstring _cardName;

	Component* _component;
};

/**
 * @class CardPanel
 * @brief A UI component that manages multiple "cards", displaying only one at a time.
 *
 * CardPanel allows for dynamic switching between different UI components,
 * each associated with a unique name (card). Only one card is visible at any time.
 */
class CardPanel : public Component
{
	/**
	 * @brief The list of all cards added to the panel.
	 */
	vector<Card> _cards;

	/**
	 * @brief The name of the currently visible card.
	 */
	wstring _visibleCardName;

protected:

	/**
	 * @brief Retrieves options for window creation.
	 *
	 * @param title Output parameter for the window title.
	 * @param widownStyles Output parameter for window style flags.
	 * @param wndClassName Output parameter for the window class name.
	 * @param isCustomWndProc Output flag indicating whether a custom window procedure is used.
	 * @return TRUE if window options were successfully retrieved, FALSE otherwise.
	 */
	BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

	/**
	 * @brief Called when the window has been successfully created.
	 */
	void windowCreated();

public:

	/**
	 * @brief Constructs a new CardPanel instance.
	 */
	CardPanel();

	/**
	 * @brief Destroys the CardPanel and its contained components.
	 */
	virtual ~CardPanel();

	/**
	 * @brief Removes all cards from the panel.
	 */
	void clear();

	/**
	 * @brief Adds a new card to the panel.
	 *
	 * @param cardName The name used to identify the new card.
	 * @param component Pointer to the component to associate with this card.
	 */
	void addCard(const wstring& cardName, Component* component);

	/**
	 * @brief Sets which card is currently visible based on its name.
	 *
	 * @param cardName The name of the card to show.
	 */
	void setVisibleCard(const wstring& cardName);

	/**
	 * @brief Renders the currently visible card.
	 *
	 * @param hDC Handle to the device context used for drawing.
	 */
	void onPaint(Graphics *g);

	/**
	 * @brief Called when the window is resized.
	 */
	void onWindowResized();

	/**
	 * @brief Called when the window is moved.
	 */
	void onWindowMoved();
};
