#pragma once

#include "Component.h"
#include "ChartLegend.h"
#include "ChartUtil.h"
#include "ToggleButton.h"

/**
 * @class Chart
 * @brief A versatile UI component for displaying various types of charts (bar, pie, line, etc.).
 *
 * Inherits from Component and implements SelectionChangeListener.
 * Supports dynamic addition of chart series, categories, and values, along with
 * optional chart type toggle buttons.
 */
class Chart : public Component,
              public SelectionChangeListener
{
	/**
	 * @brief Collection of data series to be rendered in the chart.
	 */
	vector<ChartSeries> _chartSeries;

	/**
	 * @brief List of category labels corresponding to the x-axis or other dimensions.
	 */
	vector<wstring> _categories;

	/**
	 * @brief Current chart type (e.g., ChartType::BAR, ChartType::PIE, etc.).
	 */
	ChartType _chartType;

	/**
	 * @brief Flag indicating whether chart type toggle buttons should be displayed.
	 */
	BOOL _showChartTypeToggleButtons;

	/**
	 * @brief Legend displaying series names and associated colors.
	 */
	ChartLegend _legend;

	/**
	 * @brief Toggle button to switch chart type (top position).
	 */
	ToggleButton _chartTypeToggleTop;

	/**
	 * @brief Toggle button to switch chart type (bottom position).
	 */
	ToggleButton _chartTypeToggleBottom;

	/**
	 * @brief Rectangle defining the rendering area of the chart.
	 */
	Bounds _chartRect;

	/**
	 * @brief Updates the legend with the current series and values.
	 */
	void ReloadLegendValues();

protected:

	/**
	 * @brief Retrieves window creation options such as title, style, and class name.
	 *
	 * @param title Output: the window title.
	 * @param widownStyles Output: the window style flags.
	 * @param wndClassName Output: the class name for the window.
	 * @param isCustomWndProc Output: whether a custom window procedure is used.
	 * @return TRUE if options were successfully retrieved, FALSE otherwise.
	 */
	BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

	/**
	 * @brief Called after the chart window has been created.
	 */
	void windowCreated();

public:

	/**
	 * @brief Constructs a new Chart instance.
	 */
	Chart();

	/**
	 * @brief Destroys the Chart instance.
	 */
	virtual ~Chart();

	/**
	 * @brief Sets the chart type to be displayed.
	 *
	 * @param chartType One of the CHARTTYPE_* constants.
	 */
	void setType(const ChartType& chartType);

	/**
	 * @brief Shows or hides chart type toggle buttons.
	 *
	 * @param bShow TRUE to show toggle buttons, FALSE to hide them.
	 */
	void showTypeToggleButtons(BOOL bShow);

	/**
	 * @brief Sets the categories (e.g., x-axis labels) for the chart.
	 *
	 * @param xAxisValues A list of category labels.
	 */
	void setCategories(const vector<wstring>& xAxisValues);

	/**
	 * @brief Adds a new data series to the chart.
	 *
	 * @param seriesName Name of the series.
	 * @param seriesColor Color of the series.
	 */
	void addSeries(const wstring& seriesName, COLORREF seriesColor);

	/**
	 * @brief Adds a value to an existing series.
	 *
	 * @param seriesName Name of the series to which the value should be added.
	 * @param value The value to add.
	 * @param color Optional color override for the individual value.
	 */
	void addSeriesValue(const wstring& seriesName, double value, COLORREF color);

	/**
	 * @brief Renders the chart to the given device context.
	 *
	 * @param hDC Handle to the device context used for drawing.
	 */
	void onPaint(Graphics *g);

	/**
	 * @brief Called when the window is resized.
	 */
	void onWindowResized();

	/**
	 * @brief Handles selection change events, typically triggered by user interaction.
	 *
	 * @param ev The selection change event object.
	 */
	void onSelectionChanged(const SelectionChangeEvent& ev);
};
