#pragma once

#include "Component.h"

/**
 * @class CheckBox
 * @brief A UI component that represents a checkable box with optional label text.
 *
 * Inherits from Component and allows users to toggle a boolean state (checked/unchecked).
 * Supports event listeners for selection change.
 */
class CheckBox : public Component
{
	Bounds _checkBoxRect, _textRect;

	/**
	 * @brief Indicates whether the checkbox is currently checked.
	 */
	BOOL _checked;

	/**
	 * @brief The label text displayed next to the checkbox.
	 */
	wstring _text;

protected:
	/**
	 * @brief Retrieves options for creating the checkbox window.
	 *
	 * @param title Output: window title.
	 * @param widownStyles Output: window style flags.
	 * @param wndClassName Output: class name of the window.
	 * @param isCustomWndProc Output: whether a custom window procedure is used.
	 * @return TRUE if the options were successfully retrieved, FALSE otherwise.
	 */
	BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

	/**
	 * @brief Called after the checkbox window has been successfully created.
	 */
	void windowCreated();

public:
	/**
	 * @brief Constructs a new CheckBox component.
	 */
	CheckBox();

	/**
	 * @brief Destroys the CheckBox component.
	 */
	virtual ~CheckBox();

	/**
	 * @brief Returns the preferred height
	 */
	int getPreferredHeight();

	/**
	 * @brief Sets the label text displayed next to the checkbox.
	 *
	 * @param text The text to display.
	 */
	void setText(const wstring& text);

	/**
	 * @brief Sets the checked state of the checkbox.
	 *
	 * @param checked TRUE to check the box, FALSE to uncheck.
	 */
	void setChecked(BOOL checked);

	/**
	 * @brief Returns whether the checkbox is currently checked.
	 *
	 * @return TRUE if checked, FALSE otherwise.
	 */
	BOOL isChecked();

	/**
	 * @brief Adds a listener that is notified when the selection state changes.
	 *
	 * @param l Pointer to a SelectionChangeListener to receive events.
	 */
	void addSelectionChangedListener(SelectionChangeListener* l);

public: // Overridables

	void onPaint(Graphics *g) override;
	void onWindowResized() override;

	void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed) override;
};
