#pragma once

#include "Component.h"

/**
 * @class ComboBox
 * @brief A UI component that allows users to select a single item from a dropdown list.
 *
 * This component inherits from Component and manages a list of string options.
 * It notifies registered listeners when the selected item changes.
 */
class ComboBox : public Component
{
    /**
     * @brief The list of selectable options.
     */
    vector<wstring> _selections;

    /**
     * @brief The currently selected option.
     */
    wstring _selection;

protected:
    /**
     * @brief Retrieves options used for creating the ComboBox window.
     *
     * @param title Output parameter for the window title.
     * @param widownStyles Output parameter for the window style flags.
     * @param wndClassName Output parameter for the window class name.
     * @param isCustomWndProc Output: TRUE if a custom window procedure is used.
     * @return TRUE if window options were retrieved successfully, FALSE otherwise.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the ComboBox window is successfully created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new ComboBox instance.
     */
    ComboBox();

    /**
     * @brief Destroys the ComboBox instance.
     */
    virtual ~ComboBox();

    /**
     * @brief Returns the preferred height
     */
    int getPreferredHeight();

    void clear();

    /**
     * @brief Sets the selectable options in the combo box.
     *
     * @param selections A vector of strings representing available choices.
     */
    void addItem(const wstring selection);
    BOOL getItem(int index, wstring& text);
    long getItemCount();

    /**
     * @brief Sets the selected item in the combo box.
     *
     * @param text The item text to select.
     */
    void setSelectedItem(const wstring& text);

    /**
     * @brief Gets the currently selected item in the combo box.
     *
     * @return The text of the selected item.
     */
    wstring getSelectedItem();

    /**
     * @brief Handles selection change notifications.
     *
     * @param hWndCB Handle to the combo box window.
     * @param itemIndex Index of the newly selected item.
     * @param itemText Text of the selected item.
     * @param itemChecked Unused in ComboBox (used in multi-select components).
     * @param procParams Additional context or event parameters.
     */
    void onSelectionChanged(WinHandle hWndCB, int itemIndex, const wstring& itemText, BOOL itemChecked, const ProcParams& procParams);

    /**
     * @brief Registers a listener to be notified when the selected item changes.
     *
     * @param l Pointer to a SelectionChangeListener instance.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);
};
