#pragma once

#include "Common.h"
#include "UICommon.h"
#include "EventListeners.h"
#include "Graphics.h"

class ProcParams
{
public:
	WinHandle hwnd;
	WPARAM wParam;
	LPARAM lParam;
};

/**
 * @class Component
 * @brief Base class for all UI components providing window creation, event handling, and listener management.
 *
 * This abstract class provides common functionality for windowed components, including:
 * - Unique ID generation for each component instance
 * - Window creation and management
 * - Font and appearance customization
 * - Event handling through virtual overridable methods
 * - Support for data change, selection change, action, and double-click listeners
 * - timer management
 *
 * Derived classes must implement getCreateWindowOptions() and windowCreated() to specify
 * window creation details and initialization logic.
 */
class Component
{
    /// Static counter to assign unique IDs to each Component instance.
    static long NextId;

    /// Unique identifier for this component instance.
    long _id;

    Component* _parent;

    /// If TRUE, prevents subclassing the native window procedure.
    BOOL _doNotSubclass;

    /// Command identifier associated with this component.
    long _nCommand;

    /// Registered listener for data change events.
    DataChangeListener* _dataChangeListener;

    /// Registered listener for selection change events.
    SelectionChangeListener* _selectionChangeListener;

    /// Registered listener for action events.
    ActionListener* _actionListener;

    /// Registered listener for row double-click events.
    RowDoubleClickListener* _rowDoubleClickListener;

    /// Accumulated mouse wheel delta value.
    int _mouseWheelDelta;

    /// Whether the native window has been created successfully.
    BOOL _isPaintQueued;

    /// Indicates if the component uses a custom font.
    BOOL _isCustomFont;

    /// Handle to the custom font used by the component.
    HFONT _customFont;

    /// Size of the font in points.
    long _fontSize;

    /// Name of the font used.
    wstring _fontName;

    /// Font style flags.
    BOOL _fontIsBold;
    BOOL _fontIsItalic;
    BOOL _fontIsUnderlined;

    long _lastResizeX, _lastResizeY, _lastResizeWidth, _lastResizeHeight;

    vector<NumberCollection> _scheduledTimers;

    /**
     * @brief Intercepts standard control events for the given child window handle.
     *
     * @param hWndControl Handle to the control window.
     */
    void interceptStandardControlEvents(WinHandle hWndControl);

    /**
     * @brief Creates the native window for this component.
     *
     * @param hParent Handle to the parent window.
     * @param rect The bounding rectangle for the component.
     * @param title Window title.
     * @param windowStyles Window style flags.
     * @param wndClassName Optional window class name.
     * @param isCustomWndProc TRUE if using a custom window procedure.
     * @param nCmdId Command ID associated with the window.
     * @param isVisible TRUE if the window should be visible initially.
     * @return TRUE if creation succeeded, FALSE otherwise.
     */
    BOOL createNativeWindow(Component *parent, Bounds rect, const wstring& title, UINT windowStyles,
                            const wstring& wndClassName = L"", BOOL isCustomWndProc = FALSE,
                            long nCmdId = -1, BOOL isVisible = TRUE);

protected:
    /// Handle to the native window associated with this component.
    WinHandle _hWnd;

    /// String representing the component type (e.g., "Button", "ComboBox").
    wstring _componentType;

    BOOL _showBorder;

    BOOL _isVisible;

    long _alignment;

    BOOL _lastItemIsPartiallyVisible;

    /**
     * @brief Abstract method to get window creation options.
     *
     * Derived classes must override this to provide window title, styles, class name,
     * and whether a custom window procedure is used.
     *
     * @param title Output window title.
     * @param widownStyles Output window styles.
     * @param wndClassName Output window class name.
     * @param isCustomWndProc Output TRUE if a custom window procedure is used.
     * @return TRUE if options were set successfully.
     */
    virtual BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc) = 0;

    /**
     * @brief Abstract method called after the window has been created.
     *
     * Used to perform any initialization that requires a valid window handle.
     */
    virtual void windowCreated() = 0;

    /**
     * @brief Returns the component's action command ID.
     *
     * @return Action ID associated with this component.
     */
    long getActionId();

    /**
     * @brief Marks this component's native window to avoid subclassing.
     */
    void doNotSubclass();

    /**
     * @brief Starts a timer with the given ID and interval.
     *
     * @param timerId Identifier for the timer.
     * @param milliseconds Timer interval in milliseconds.
     */
    void startTimer(unsigned int timerId, unsigned int milliseconds);

    /**
     * @brief Stops the timer identified by timerId.
     *
     * @param timerId Identifier of the timer to stop.
     */
    void stopTimer(unsigned int timerId);

public:
    /**
     * @brief Constructs a new Component instance.
     */
    Component();

    /**
     * @brief Virtual destructor.
     */
    virtual ~Component();

    /**
     * @brief Gets the unique ID assigned to this component.
     *
     * @return Unique component ID.
     */
    long getId();

    void setParent(Component* parent);
    Component* getParent();
    Component* getTopLevelParent();

    void bringWindowToTop();

    /**
     * @brief Gets the component type string.
     *
     * @return Component type name.
     */
    wstring getComponentType();

    /**
     * @brief Returns whether the native window has been created.
     *
     * @return TRUE if the window is created, FALSE otherwise.
     */
    BOOL isWindowCreated();

    /**
     * @brief Returns the native WinHandle for this component.
     *
     * @return Handle to the native window.
     */
    WinHandle getWindowHandle();

    /**
     * @brief Creates the component's native window as a child of hParent.
     *
     * @param hParent Parent window handle.
     * @param rect Initial window rectangle.
     * @param nCmd Optional command ID associated with the component.
     * @param isVisible TRUE to show the window immediately.
     * @return TRUE if creation succeeded.
     */
    int createComponent(Component* parent, Bounds rect, long nCmd = -1, BOOL isVisible = TRUE);

	void setFocus();

    /**
     * @brief Sets the font for the component.
     *
     * @param fontSize Font size in points.
     * @param isBold TRUE to make font bold.
     * @param isItalic TRUE to make font italic.
     * @param isUnderlined TRUE to underline the font.
     * @param fontName Name of the font family.
     */
    void setFont(long fontSize, BOOL isBold = FALSE, BOOL isItalic = FALSE, BOOL isUnderlined = FALSE, const wstring& fontName = L"Arial");

    BOOL getCustomFont(long& fontSize, BOOL& isBold, BOOL& isItalic, BOOL& isUnderlined, wstring& fontName);

    void showBorder(BOOL bShow = TRUE);

    void setAlignment(long alignment/*-1, 0, 1*/);

    /**
     * @brief Enables or disables the component.
     *
     * @param bEnable TRUE to enable, FALSE to disable.
     */
    void setEnabled(BOOL bEnable = TRUE);

    /**
     * @brief Shows or hides the component.
     *
     * @param bShow TRUE to show, FALSE to hide.
     */
    void setVisible(BOOL bShow = TRUE);

    void GetClientRect(Bounds& rect);
    void GetWindowRect(Bounds& rect);

    /**
     * @brief Moves and resizes the component window.
     *
     * @param x New X coordinate.
     * @param y New Y coordinate.
     * @param width New width.
     * @param height New height.
     */
    void moveWindow(long x, long y, long width, long height, BOOL bRedraw = TRUE);

    /**
     * @brief Sets the accumulated mouse wheel delta.
     *
     * @param delta Mouse wheel delta value.
     */
    void setMouseWheelDelta(int delta);

    /**
     * @brief Gets the current mouse wheel delta.
     *
     * @return Mouse wheel delta value.
     */
    int getMouseWheelDelta();

    /**
     * @brief Returns the minimum height the component can have.
     *
     * @return Minimum height in pixels.
     */
    virtual int getMinimumHeight();

    /**
     * @brief Returns the preferred height of the component.
     *
     * @return Preferred height in pixels.
     */
    virtual int getPreferredHeight();

    /**
     * @brief Requests the component to repaint itself.
     */
    void repaint(Component* comp = NULL, BOOL repaintImmediately = FALSE); // Must be defaulted to FALSE

    /**
     * @brief clears the queued paint request. This function should only be called internally from the library.
     */
    void clearQueuedPaintRequest();

    virtual BOOL getChildren(vector<Component*>& children) { return FALSE; };

public: // Overridable event handlers

    /** @brief Called to paint the component's client area. */
    virtual void onPaint(Graphics *g) {}

    /** @brief Called when the component's window is resized. */
    virtual void onWindowResized() {}

    /** @brief Called when the component's window is moved. */
    virtual void onWindowMoved() {}

    /**
     * @brief Called when an action occurs on the component.
     *
     * @param hTarget Handle to the target window.
     * @param actionId Action identifier.
     * @param procParams Additional parameters.
     */
    virtual void onAction(WinHandle hTarget, long actionId, const ProcParams& procParams) {}

    /**
     * @brief Called when the selection changes.
     *
     * @param hTarget Handle to the target window.
     * @param itemIndex Index of the selected item.
     * @param itemText Text of the selected item.
     * @param itemChecked Checked state (for multi-select components).
     * @param procParams Additional parameters.
     */
    virtual void onSelectionChanged(WinHandle hTarget, int itemIndex, const wstring& itemText, BOOL itemChecked, const ProcParams& procParams) {}

    /**
     * @brief Called when data changes in the component.
     *
     * @param hTarget Handle to the target window.
     * @param newValue The new data value.
     * @param procParams Additional parameters.
     */
    virtual void onDataChanged(WinHandle hTarget, const wstring& newValue, const ProcParams& procParams) {}

//    virtual BOOL onDrawMeasureItem(LPMEASUREITEMSTRUCT dis, const ProcParams& procParams) { return FALSE; }
//    virtual BOOL onDrawItem(LPDRAWITEMSTRUCT dis, const ProcParams& procParams) { return FALSE; }

    /** @brief Called when the mouse is pressed within the component. */
    virtual void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed) {}

    /** @brief Called when the mouse button is released within the component. */
    virtual void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed) {}

    /** @brief Called when the right mouse button is clicked within the component. */
    virtual void onMouseRightClicked(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed) {}

    /** @brief Called when the mouse is moved within the component. */
    virtual void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed) {}

    /**
     * @brief Called when the mouse wheel is used over the component.
     *
     * @param hWnd Handle to the window receiving the event.
     * @param x X coordinate of the mouse.
     * @param y Y coordinate of the mouse.
     * @param delta Mouse wheel delta.
     * @return TRUE if the event was handled, FALSE otherwise.
     */
    virtual BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta) { return FALSE; }

    // Keyboard key event handlers (shiftPressed and ctrlPressed indicate modifier key state)
    virtual void onArrowLeft(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onArrowRight(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onPageUp(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onPageDown(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyInsert(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyDelete(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyBackSpace(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyTyped(wchar_t ch, BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyTab(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyEnter(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF1(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF2(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF3(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF4(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF5(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF6(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF7(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF8(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF9(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF10(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF11(BOOL shiftPressed, BOOL ctrlPressed) {}
    virtual void onKeyF12(BOOL shiftPressed, BOOL ctrlPressed) {}

    /** @brief Called when the component gains keyboard focus. */
    virtual void onFocusGained() {}

    /** @brief Called when the component loses keyboard focus. */
    virtual void onFocusLost() {}

    /**
     * @brief Called on timer events.
     *
     * @param timerId The timer ID.
     */
    virtual void onTimer(unsigned int timerId) {}


protected: // Listener management

    /**
     * @brief Adds a data change listener.
     *
     * @param l Pointer to a DataChangeListener instance.
     */
    void protectedAddDataChangedListener(DataChangeListener* l);

    /**
     * @brief Adds a selection change listener.
     *
     * @param l Pointer to a SelectionChangeListener instance.
     */
    void protectedAddSelectionChangedListener(SelectionChangeListener* l);

    /**
     * @brief Adds an action listener.
     *
     * @param l Pointer to an ActionListener instance.
     */
    void protectedAddActionListener(ActionListener* l);

    /**
     * @brief Adds a row double-click listener.
     *
     * @param l Pointer to a RowDoubleClickListener instance.
     */
    void protectedAddItemDoubleClickedListener(RowDoubleClickListener* l);

    /**
     * @brief Fires a data changed event to registered listeners.
     *
     * @param oldValue Previous data value.
     * @param newValue New data value.
     */
    void protectedFireDataChangedEvent(const wstring& oldValue, const wstring& newValue);

    /**
     * @brief Fires a selection changed event to registered listeners.
     *
     * @param selectionIndex Index of the new selection.
     * @param selectionValue Text of the new selection.
     * @param checked Checked state (for multi-select).
     */
    void protectedFireSelectionChangedEvent(long selectionIndex = -1, const wstring& selectionValue = L"", BOOL checked = FALSE);

    /**
     * @brief Fires an action event to registered listeners.
     *
     * @param actionId Action identifier.
     * @param actionName Name of the action.
     */
    void protectedFireActionEvent(long actionId = -1, const wstring& actionName = L"");

    /**
     * @brief Fires a row double-click event to registered listeners.
     *
     * @param rowIndex Index of the double-clicked row.
     */
    void protectedFireItemDoubleClickedEvent(__int64 rowIndex);
};
