#pragma once

#include "Component.h"
#include "Calendar.h"

/**
 * @class DateTimePicker
 * @brief A UI component for selecting and displaying dates.
 *
 * Inherits from Component and implements a date picker control with a simple
 * string-based interface for setting and getting the selected date in the format "yyyy-mm-dd".
 */
class DateTimePicker : public Component,
                        public SelectionChangeListener
{
    /// Currently selected date in "yyyy-mm-dd" format.
    wstring _date;

    Calendar _calendar;

protected:
    /**
     * @brief Provides window creation options specific to the DateTimePicker control.
     *
     * Overrides Component::getCreateWindowOptions to supply the window title,
     * styles, window class name, and whether a custom window procedure is used.
     *
     * @param title Output window title.
     * @param widownStyles Output window style flags.
     * @param wndClassName Output window class name.
     * @param isCustomWndProc Output TRUE if a custom window procedure will be used.
     * @return TRUE if options are set successfully.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc) override;

    /**
     * @brief Called after the native window has been created.
     *
     * Used to perform any initialization that requires the window handle.
     */
    void windowCreated() override;

public:
    /**
     * @brief Constructs a new DateTimePicker component.
     */
    DateTimePicker();

    /**
     * @brief Destructor.
     */
    virtual ~DateTimePicker();

    /**
     * @brief Returns the preferred height
     */
    int getPreferredHeight();

    /**
     * @brief Sets the selected date.
     *
     * The date must be provided as a string in the format "yyyy-mm-dd".
     *
     * @param dateTime The date string to set.
     */
    void setSelectedDate(const wstring& dateTimeYYYYMMDD);

    /**
     * @brief Retrieves the currently selected date.
     *
     * @return The selected date string in "yyyy-mm-dd" format.
     */
    wstring getSelectedDate();

    /**
     * @brief Renders the CheckComboBox.
     *
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Called when the window containing the control is resized.
     */
    void onWindowResized();

    /**
     * @brief Called when the window containing the control is moved.
     */
    void onWindowMoved();

    /**
     * @brief Handles mouse press events.
     *
     * @param hWnd Handle to the window receiving the event.
     * @param x X-coordinate of the mouse event.
     * @param y Y-coordinate of the mouse event.
     * @param clickCount Number of clicks (e.g., single or double).
     * @param shiftPressed Whether Shift was pressed during the click.
     * @param ctrlPressed Whether Ctrl was pressed during the click.
     */
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Responds to selection changes triggered by internal components like CheckList.
     *
     * @param ev The selection change event containing details of the change.
     */
    void onSelectionChanged(const SelectionChangeEvent& ev);

    /**
     * @brief Adds a listener to be notified when the selection changes.
     *
     * @param l Pointer to a SelectionChangeListener to notify.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);
};
