#pragma once

#include "Component.h"

class SearchDialogColumn
{
public:

    wstring columnName;
    wstring columnTitle;
    long columnWidthRatio;
    BOOL columnVisible;
};
/**
 * @class DialogFactory
 * @brief A utility class providing static methods to show common dialog boxes.
 *
 * This factory class offers convenient static methods to display error, warning,
 * info, and success messages, as well as dialogs for file selection, folder selection,
 * color picking, font picking, printer setup, confirmation prompts, and generic input dialogs.
 */
class DialogFactory
{
public:
    /**
     * @brief Shows an error message dialog.
     *
     * @param parent Handle to the parent window.
     * @param errorMessage The error message to display.
     * @param title Optional dialog title. Defaults to an empty string.
     */
    static void showError(Component *parent, const wstring& errorMessage, const wstring& title = L"");

    /**
     * @brief Shows a warning message dialog.
     *
     * @param parent Handle to the parent window.
     * @param warningMessage The warning message to display.
     * @param title Optional dialog title. Defaults to an empty string.
     */
    static void showWarning(Component *parent, const wstring& warningMessage, const wstring& title = L"");

    /**
     * @brief Shows an informational message dialog.
     *
     * @param parent Handle to the parent window.
     * @param infoMessage The informational message to display.
     * @param title Optional dialog title. Defaults to an empty string.
     */
    static void showInfo(Component *parent, const wstring& infoMessage, const wstring& title = L"");

    /**
     * @brief Shows a success message dialog.
     *
     * @param parent Handle to the parent window.
     * @param successMessage The success message to display.
     * @param title Optional dialog title. Defaults to an empty string.
     */
    static void showSuccess(Component *parent, const wstring& successMessage, const wstring& title = L"");

    /**
     * @brief Shows a confirmation dialog with Yes/No options.
     *
     * @param parent Handle to the parent window.
     * @param promptMessageText The prompt message to display.
     * @param title Optional dialog title. Defaults to an empty string.
     * @return TRUE if user confirms (Yes), FALSE otherwise.
     */
    static BOOL showConfirmDialog(Component *parent, const wstring& promptMessageText, const wstring& title = L"");

    /**
     * @brief Shows an Open File dialog to select a file.
     *
     * @param hWndParent Handle to the parent window.
     * @param selection Output parameter receiving the selected file path.
     * @return TRUE if a file was selected, FALSE otherwise.
     */
    static BOOL showOpenFileDialog(Component *parent, vector<wstring>& selection, BOOL allowMultiSelection = FALSE);

    /**
     * @brief Shows a Save File dialog to specify a file to save.
     *
     * @param hWndParent Handle to the parent window.
     * @param selection Output parameter receiving the selected file path.
     * @return TRUE if a file path was selected, FALSE otherwise.
     */
    static BOOL showSaveFileDialog(Component *parent, wstring& selection);

    /**
     * @brief Shows a folder selection dialog.
     *
     * @param hWndParent Handle to the parent window.
     * @param selection Output parameter receiving the selected folder path.
     * @return TRUE if a folder was selected, FALSE otherwise.
     */
    static BOOL showFolderSelectionDialog(Component *parent, wstring& selection);

    /**
     * @brief Shows a color selection dialog.
     *
     * @param hWndParent Handle to the parent window.
     * @param selection Input/output COLORREF specifying initial and receiving selected color.
     * @return TRUE if a color was selected, FALSE otherwise.
     */
    static BOOL showColorSelectionDialog(Component *parent, COLORREF& selection);

    /**
     * @brief Shows a font selection dialog.
     *
     * @param hWndParent Handle to the parent window.
     * @param fontName Input/output font name string.
     * @param fontHeight Input/output font height.
     * @param fontWeight Input/output font weight.
     * @param isItalic Input/output italic flag.
     * @param color Input/output COLORREF for font color.
     * @return TRUE if a font was selected, FALSE otherwise.
     */
    static BOOL showFontSelectionDialog(Component *parent,
                                        wstring& fontName,
                                        long& fontHeight,
                                        long& fontWeight,
                                        BOOL& isItalic,
                                        COLORREF& color);

    /**
     * @brief Shows a printer selection dialog.
     *
     * @param hWndParent Handle to the parent window.
     * @return Handle to the device context (DCHandle) for the selected printer.
     *         Caller must call DeleteDC on the returned DCHandle when done.
     */
    static DCHandle showPrinterSelectionDialog(Component* parent, Bounds& printerRect);

    /**
     * @brief Shows a printer setup dialog to configure margins.
     *
     * @param hWndParent Handle to the parent window.
     * @param leftMargin Input/output left margin.
     * @param topMargin Input/output top margin.
     * @param rightMargin Input/output right margin.
     * @param bottomMargin Input/output bottom margin.
     * @return TRUE if setup was successful, FALSE otherwise.
     */
    static BOOL showPrinterSetupDialog(Component *parent,
                                      long& leftMargin, long& topMargin,
                                      long& rightMargin, long& bottomMargin);

    /**
     * @brief Shows a generic input dialog with multiple labeled fields.
     *
     * @param parent Handle to the parent window.
     * @param labels Vector of label strings for input fields.
     * @param values Input/output vector of values corresponding to labels.
     * @return TRUE if user confirmed input, FALSE if cancelled.
     */
    static BOOL showInputDialog(Component *parent, const vector<wstring>& labels, vector<wstring>& values);

    /**
     * @brief Shows a generic input dialog with multiple labeled fields.
     *
     * @param parent Handle to the parent window.
     * @param selectedDate Input/Output selected date.
     * @return TRUE if user confirmed input, FALSE if cancelled.
     */
    static BOOL showDateInputDialog(Component *parent, wstring& selectedDate);

    /**
     * @brief Shows a generic record search dialog with search capability
     *
     * @param parent Handle to the parent window.
     * @param title Dialog title.
     * @param searchFieldNames Array of search field names.
     * @param pager A class that implements the Pager interface, usually this would be the parent passed to this function.
     * @param pagerRequestSourceComponentId A custom id used to identify the source component calling the GetPage on the pager object.
    */
    static BOOL showRecordSearchDialog(Component* parent, const wstring& title, 
        const vector<wstring>& searchFieldNames,
        const vector<SearchDialogColumn>& gridColumns,
        Pager* pager, long pagerRequestSourceComponentId, 
        const wstring& idColumnName, wstring& selectedRecordId);
};

