#pragma once

#include "Component.h"
#include "GridPanel.h"
#include "Button.h"
#include "Label.h"
#include "TitleBar.h"
#include "MenuBar.h"

class StatusBar : public GridPanel, public ActionListener
{
    Label _textLabel;
    Button _licenseBtn;
    Button _themeBtn;

    BOOL _showThemeToggler;

    wstring _licenseMessage;

public:

    StatusBar();
    virtual ~StatusBar();

    void setText(const wstring& text);
    void setLicenseMessage(const wstring& message);

    void setShowThemeToggler(BOOL bShow = TRUE);

    void onAction(const ActionEvent& ev);
};

/**
 * @class Frame
 * @brief A top-level window frame component with licensing support and content pane.
 *
 * The Frame class represents a window frame that contains a GridPanel as its content pane.
 * It supports licensing validation, window events, and timer handling.
 */
class Frame: public Component
{
    GridPanel _contentPane;       ///< Main content pane of the frame
    StatusBar _statusBar;
    TitleBar  _titleBar;
    MenuBar   _menuBar;
    BOOL      _menuBarVisible;

    wstring _title;               ///< Title text of the frame window
    wstring _statusBarText;
    wstring _ltoken;
    BOOL _lValid;
    wstring _lTo;
    wstring _lDate;
    wstring _lApplication;
    BOOL _isCentered;

    BOOL _showThemeToggler;

    ActionListener* _menuBarListener;

    Bounds _bannerRect;

    /**
     * @brief Internal method for license validation or client-specific logic.
     * @param t Input token or string for processing.
     */
    void _clt(const wstring& t);

protected:

    /**
     * @brief Retrieves window creation options such as title, styles, and class name.
     * @param title Output parameter for window title.
     * @param widownStyles Output parameter for window styles.
     * @param wndClassName Output parameter for window class name.
     * @param isCustomWndProc Output flag indicating custom window procedure usage.
     * @return TRUE if options retrieved successfully; otherwise FALSE.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the native window has been created.
     */
    void windowCreated();

protected:

    /**
     * @brief Returns a pointer to the frame's content pane (GridPanel).
     * @return Pointer to GridPanel content pane.
     */
    GridPanel *getContentPane();

    void setWindowCentered();

    void setShowThemeToggler(BOOL bShow = TRUE);

public:

    /**
     * @brief Constructs a Frame with specified window class name and title.
     * @param wndClassName The window class name.
     * @param title The window title.
     */
    Frame(const wstring& title);

    /**
     * @brief Destructor.
     */
    virtual ~Frame();

    void setIcon(const IconSource& iconSource);

    void setTitle(const wstring& title);

    void setStatusBarText(const wstring& text);

    BOOL usesCustomCaption();

    int show(long x = -1, long y = -1, long width = -1, long height = -1);

    /**
     * @brief Sets the license token and updates licensing status.
     * @param licenseToken License token string.
     */
    void setLicense(const wstring& licenseToken);

    void setMenuBarListener(ActionListener* l);

    void Close();

    Menu* addMenu(const wstring& menuName);
    void setMenuListener(ActionListener* l);

    /**
     * @brief Abstract method called to perform window construction tasks.
     *
     * Must be implemented by subclasses.
     */
    virtual void onConstructWindow() = 0;

    /**
     * @brief Handles timer events.
     * @param timerId Identifier of the timer.
     */
    void onTimer(unsigned int timerId);

    void onPaint(Graphics* g);
    void onWindowResized();

    /**
     * @brief Handles window move events.
     */
    void onWindowMoved();

    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);

    void onAction(WinHandle hTarget, long actionId, const ProcParams& procParams);
};
