#pragma once

#include "PanelBase.h"
#include "Scrollbar.h"

class GridPanelColumn
{
public:
    GridPanelColumn(int width)
    {
        _width = width;
        _calculatedWidth = width;
    }

    int _width;            
    int _calculatedWidth;  
};

class GridPanelRow
{
public:
    GridPanelRow(int height)
    {
        _height = height;
        _calculatedHeight = height;
    }

    int _height;            
    int _calculatedHeight;  
};

/**
 * @class GridPanel
 * @brief A panel that arranges child components in a grid layout.
 * 
 * Supports flexible and fixed column widths and row heights, scrolling,
 * and mouse wheel support.
 */
class GridPanel : public PanelBase, public ScrollbarListener
{
    Scrollbar _scrollBar;

    vector<GridPanelColumn> _columns; ///< Columns managed by the panel
    vector<GridPanelRow> _rows;       ///< Rows managed by the panel

    long _fillRowsMinHeight;          ///< Minimum allowed row height for fill rows

    long _margin;                     ///< Sets the left, top, right, and bottom margins
    long _vGap;                     ///< Sets the gap between rows
    long _allRowsHeight;                ///< Total height of all rows combined
    long _rowHeight;                    ///< Default scrolling row height (if applicable)
    long _totalRowCount;

    long _maxVisibleRows;             ///< Maximum number of visible rows
    long _firstVisibleRow;            ///< Index of the first visible row

    long _borderWidthLeft, _borderWidthTop, _borderWidthRight, _borderWidthBottom;

    /**
     * @brief Recalculates the layout for all columns and rows.
     */
    void Layout();

protected:
    /**
     * @brief Returns the class name of the component.
     * @return The class name as a wstring.
     */
    wstring getClassName();

    /**
     * @brief Called after the native window is created.
     */
    void windowCreated();

public:
    GridPanel();
    virtual ~GridPanel();

    /**
     * @brief Clears all rows, columns, and child components from the panel.
     */
    void clear();

    /**
     * @brief Sets the left, top, right, and bottom margins
     *
     * @param margin margin width/height.
     */
    void setMargin(int margin);

    void setBorder(long borderWidthLeft, long borderWidthTop, long borderWidthRight, long borderWidthBottom);

    /**
     * @brief Sets the layout structure by defining columns and rows sizes.
     * 
     * Use -1 for flexible sizes that adapt to available space.
     * Example: setLayout({-1, 100, -1}, {30, 30, 30, -1, 30});
     * 
     * @param columns Vector of column widths.
     * @param rows Vector of row heights.
     */
    void setLayout(const vector<int>& columns, const vector<int>& rows);

    /**
     * @brief Sets the minimum allowed height for fill rows
     */
    void setFillRowsMinHeight(int minHeight);

    /**
     * @brief Adds a child component at a specific grid location.
     * 
     * @param component Pointer to the component to add.
     * @param column Column index where the component will be placed.
     * @param row Row index where the component will be placed.
     * @param columnCount Number of columns the component spans (default is 1).
     * @param rowCount Number of rows the component spans (default is 1).
     */
    void addComponent(Component* component, int column, int row, int columnCount = 1, int rowCount = 1);

    /**
     * @brief Called when the window is resized; recalculates layout.
     */
    void onWindowResized();

    /**
     * @brief Paints the panel and its child components.
     * 
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
