#pragma once

#include "Component.h"

#include "CardPanel.h"
#include "GridPanel.h"
#include "FluidPanel.h"
#include "Grid.h"
#include "Button.h"

/**
 * @class GridView
 * @brief A complex component combining a grid and form view with CRUD capabilities.
 * 
 * The GridView supports both inline editing and form-based editing of grid rows,
 * switching between a grid display and a detailed form view.
 * It also provides buttons for adding, editing, deleting, saving, and canceling changes.
 * Listeners can be registered to react to CRUD events.
 * 
 * Implements ActionListener to handle button actions and RowDoubleClickListener
 * to handle double-click events on grid rows.
 */
class GridView : public Component,
                 public ActionListener,
                 public RowDoubleClickListener
{
    CardPanel _cardPanel;      ///< Container to switch between grid and form views
    /*GridPanel*/FluidPanel _formView; ///< Form view for editing row details
    Grid _gridView;            ///< Grid displaying rows and columns

    Button _buttonAdd;         ///< Button to add a new row
    Button _buttonEdit;        ///< Button to edit the selected row
    Button _buttonDelete;      ///< Button to delete the selected row
    Button _buttonSave;        ///< Button to save changes in the form
    Button _buttonCancel;      ///< Button to cancel editing

    vector<Component*> _formComponents; ///< Components used in the form view

    GridViewListener* _listener; ///< Listener for CRUD events

    long _editingRowIndex;      ///< Currently editing row index
    string _editingType;        ///< Type of editing ("inline", "form", etc.)
    BOOL _inlineEditingEnabled; ///< Whether inline editing is enabled

    /**
     * @brief Creates child components within the given parent window handle.
     * @param hParent The handle of the parent window.
     */
    void createChildren(Component *parent);

    /**
     * @brief Loads data into the form view from the current editing row.
     */
    void loadForm();

    /**
     * @brief Copies data from the currently selected grid row to the form components.
     */
    void gridRowToForm();

    /**
     * @brief Copies data from the form components back to the grid row.
     * @return TRUE if the form data was successfully applied to the grid row, FALSE otherwise.
     */
    BOOL formToGridRow();

protected:
    /**
     * @brief Provides window creation options specific to this component.
     * @param title Window title string (output).
     * @param widownStyles Window style flags (output).
     * @param wndClassName Window class name string (output).
     * @param isCustomWndProc TRUE if a custom window procedure is used (output).
     * @return TRUE if options were successfully provided.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the native window is created; initializes child components.
     */
    void windowCreated();

public:

    BOOL getChildren(vector<Component*>& children);

public:
    GridView();
    virtual ~GridView();

    /**
     * @brief Clears all data and resets the grid and form views.
     */
    void clear();

    /**
     * @brief Enables or disables inline editing mode.
     * @param bEnable TRUE to enable inline editing, FALSE to disable.
     */
    void enableInlineEditing(BOOL bEnable);

    __int64 getRowCount();

    /**
     * @brief Adds a column to the grid.
     * @param title Column header title.
     * @param width Column width in pixels.
     * @param alignment Text alignment (-1: left, 0: center, 1: right).
     * @param editStyle Editing style (e.g., EDIT_TEXTFIELD).
     * @param editOptions Optional list of edit options (for combo boxes etc.).
     */
    void addColumn(const wstring& name, const wstring& title, BOOL isVisible, UINT width, long alignment = -1, BOOL isSortable = FALSE, BOOL isFilterable = FALSE,
        BOOL isEditable = FALSE, const EditType& editStyle = EditType::TEXTFIELD, const vector<wstring>& editOptions = {});

    /**
     * @brief Adds a new empty row to the grid.
     * @return The index of the newly added row.
     */
    long addRow();

    /**
     * @brief Sets the value of a specific cell in the grid.
     * @param rowIndex Index of the row.
     * @param columnIndex Index of the column.
     * @param value The value to set.
     */
    void setCellValue(long rowIndex, long columnIndex, const wstring& value);
    void setCellValue(long rowIndex, const wstring& columnName, const wstring& value);

    /**
     * @brief Gets the value of a specific cell in the grid.
     * @param rowIndex Index of the row.
     * @param columnIndex Index of the column.
     * @return The value stored in the cell.
     */
    wstring getCellValue(long rowIndex, long columnIndex);
    wstring getCellValue(long rowIndex, const wstring& columnName);

    /**
     * @brief Sets the listener for GridView CRUD events.
     * @param listener Pointer to an object implementing GridViewListener.
     */
    void setListener(GridViewListener* listener);

    /**
     * @brief Called when the window is resized; adjusts layouts.
     */
    void onWindowResized();

    /**
     * @brief Called to paint the GridView component.
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Handles actions triggered by buttons or other components.
     * @param ev The action event describing the source and action.
     */
    void onAction(const ActionEvent& ev);

    /**
     * @brief Handles double-click events on grid rows.
     * @param ev The double-click event.
     */
    void onItemDoubleClicked(const RowDoubleClickEvent& ev);
};
