#pragma once

#include "Component.h"
#include "Scrollbar.h"

class ListBoxItem
{
public:
    wstring _text;
    COLORREF _color;
    Bounds _rect;
};

/**
 * @class ListBox
 * @brief A UI component that displays a list of selectable string options.
 * 
 * The ListBox allows the user to select an item from a list.
 * It supports selection change notifications via listeners.
 */
class ListBox : public Component, public ScrollbarListener
{
    Scrollbar _scrollBar;

    std::vector<ListBoxItem> _selections;  ///< Available selectable options.

    long _maxVisibleRows;         ///< Maximum rows visible in the current view
    __int64 _firstVisibleRow;        ///< Index of the first visible row
    __int64 _lastVisibleRow;         ///< Index of the last visible row
    __int64 _currentRow;             ///< Index of the currently selected row

    BOOL _isColorList;

    /**
     * @brief Whether the listbox is shown as a popup.
     */
    BOOL _isPopup;
    BOOL _hidePopupOnFocusLoss;

protected:

    /**
     * @brief Provides native window creation options for the ListBox control.
     * 
     * @param title Window title (usually unused for list boxes).
     * @param widownStyles Window styles flags.
     * @param wndClassName Window class name.
     * @param isCustomWndProc Whether to use a custom window procedure.
     * @return TRUE if options successfully provided.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called once the native window has been created.
     */
    void windowCreated();

public:

    /**
     * @brief Constructs an empty ListBox component.
     */
    ListBox();

    /**
     * @brief Destructor.
     */
    virtual ~ListBox();

    int getPreferredHeight();

    /**
     * @brief Sets whether the listbox is used in a popup context.
     *
     * @param isPopup TRUE to mark the listbox as a popup, FALSE otherwise.
     */
    void setAsPopup(BOOL isPopup, BOOL hidePopupOnFocusLoss = TRUE);

    /**
     * @brief Return TRUE if this component is displayed as a popup (e.g. dropdown from CheckComboBox).
     */
    BOOL isPopup();

    void setColorsListType();

    void clear();

    /**
     * @brief Adds to the list of selectable options.
     * 
     * @param text The text of the item
     * @param color The color value of the item in case the setColorsListType is called
     */
    void addItem(const wstring& text, COLORREF color = Theme::BackgroundColor);
    BOOL getItem(int index, wstring& text);
    long getItemCount();

    /**
     * @brief Sets the currently selected item by text.
     * 
     * @param text The item to select.
     */
    void setSelectedItem(const wstring& text);

    /**
     * @brief Retrieves the currently selected item text.
     * 
     * @return The selected item string.
     */
    wstring getSelectedItem();

public: // Overridables

    void onPaint(Graphics* g) override;
    void onWindowResized() override;

    void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onPageUp(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onPageDown(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed) override;
    void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed) override;

    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed) override;
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta) override;

    void onFocusGained() override;
    void onFocusLost() override;

public:

    void addSelectionChangedListener(SelectionChangeListener* l);
    void addItemDoubleClickedListener(RowDoubleClickListener* l);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
