#pragma once

#include "Component.h"
#include "Scrollbar.h"

class RadioItem
{
public:
    RadioItem(const wstring& actionName, const wstring& text)
    {
        _action = actionName;
        _text = text;
    }

    wstring _action; 
    wstring _text;   
    Bounds _rect;      
};

/**
 * @brief A radio button group component supporting multiple selectable options.
 */
class Radio : public Component, public ScrollbarListener
{
    Scrollbar _scrollBar;

    vector<RadioItem> _items;       /**< List of radio options */
    long _selectedIndex;             /**< Index of currently selected option */

    wstring _orientation;            /**< Layout orientation: "VERT" (vertical) or "HORZ" (horizontal) */

    long _maxVisibleItems;           /**< Maximum number of visible items */
    __int64 _firstVisibleItem;          /**< Index of first visible item */
    __int64 _lastVisibleItem;           /**< Index of last visible item */

protected:
    /**
     * @brief Gets window creation options.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called when the window is created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new Radio component.
     */
    Radio();

    /**
     * @brief Destroys the Radio component.
     */
    virtual ~Radio();

    int getPreferredHeight();

    /**
     * @brief Removes all radio options.
     */
    void clear();

    /**
     * @brief Sets the orientation of the radio group.
     * 
     * @param orientation "VERT" for vertical layout, "HORZ" for horizontal layout (default is "VERT").
     */
    void setOrientation(const wstring& orientation = L"VERT");

    /**
     * @brief Gets the current orientation.
     * 
     * @return wstring Current orientation ("VERT" or "HORZ").
     */
    wstring getOrientation();

    /**
     * @brief Adds a radio option.
     * 
     * @param actionName Internal action name identifying this option.
     * @param text Display text for the option.
     */
    void addOption(const wstring& actionName, const wstring& text);

    /**
     * @brief Sets the selected radio option by its action name.
     * 
     * @param action The action name to select.
     */
    void setSelectedtem(const wstring& action);

    /**
     * @brief Gets the action name of the currently selected option.
     * 
     * @return wstring The selected option's action name.
     */
    wstring getSelectedtem();

public: // Overridables

    /**
     * @brief Paints the control.
     */
    void onPaint(Graphics *g);

    /**
     * @brief Handles window resize event.
     */
    void onWindowResized();

    /**
     * @brief Handles Up arrow key press.
     */
    void onArrowUp(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles Down arrow key press.
     */
    void onArrowDown(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles Page Up key press.
     */
    void onPageUp(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles Page Down key press.
     */
    void onPageDown(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles Home key press.
     */
    void onKeyHome(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles End key press.
     */
    void onKeyEnd(BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles mouse press events.
     */
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseMoved(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);

    /**
     * @brief Handles mouse wheel scrolling.
     * 
     * @return TRUE if handled, FALSE otherwise.
     */
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);

    /**
     * @brief Adds a listener for selection change events.
     * 
     * @param l Pointer to a SelectionChangeListener.
     */
    void addSelectionChangedListener(SelectionChangeListener* l);

public:

    void scrollBarActivity(__int64 firstVisibleItem);
    void scrollBarRepaint();
};
