#pragma once

#include "Common.h"

/**
 * @brief Represents a UI theme, managing brushes, fonts, and colors.
 * 
 * Provides access to commonly used brushes and fonts, and manages color
 * definitions for various UI components.
 */
class Theme
{
    HBRUSH _hBackgroundBrush;      /**< Brush for background painting */
    HBRUSH _hLabelBrush;           /**< Brush for label text */
    HBRUSH _hInputFieldBrush;      /**< Brush for input fields */
    HBRUSH _hBtnBrush;             /**< Brush for buttons */

    HFONT _hDefaultFont;           /**< Default font handle */

    wstring _currentTheme;

    wstring _savedThemeFilePath;

public:
    /**
     * @brief Constructs a new Theme object.
     */
    Theme();

    /**
     * @brief Destructor.
     */
    virtual ~Theme();

    /**
     * @brief Returns a singleton instance of Theme.
     * 
     * @return Pointer to the Theme instance.
     */
    static Theme* GetInstance();

    /**
     * @brief Creates or retrieves a background brush.
     * 
     * @param hDC Handle to device context.
     * @return Handle to the background brush.
     */
    HBRUSH createBackgroundBrush(DCHandle hDC);

    /**
     * @brief Creates or retrieves a label brush.
     * 
     * @param hDC Handle to device context.
     * @return Handle to the label brush.
     */
    HBRUSH createLabelBrush(DCHandle hDC);

    /**
     * @brief Creates or retrieves an input field brush.
     * 
     * @param hDC Handle to device context.
     * @return Handle to the input field brush.
     */
    HBRUSH createInputFieldBrush(DCHandle hDC);

    /**
     * @brief Creates or retrieves a button brush.
     * 
     * @param hDC Handle to device context.
     * @return Handle to the button brush.
     */
    HBRUSH createButtonBrush(DCHandle hDC);

    /**
     * @brief These functions are used to set the application theme - Must be called from the constructor of the Frame class
     */
    void setDefaultTheme();
    void setDarkTheme();
    void setSilverTheme();
    void setTaupeTheme();
    void setGoldTheme();
    void setPurpleTheme();

    void toggleTheme();

    void saveTheme();
    void loadTheme();

    void ButtonTypeToColors(const ButtonType& btnType, COLORREF& startBkColor, COLORREF& endBkColor, COLORREF& color);

    /**
     * @brief Sets the default font for a window.
     * 
     * @param hWnd Handle to the window.
     */
    void setDefaultFont(WinHandle hWnd);

    /**
     * @brief Sets the font for a window with specified properties.
     * 
     * @param hWnd Handle to the window.
     * @param fontSize Desired font size.
     * @param isBold TRUE to use bold font, default FALSE.
     * @param isItalic TRUE to use italic font, default FALSE.
     * @param isUnderlined TRUE to use underlined font, default FALSE.
     * @param fontName Name of the font face, default "Arial".
     * @return Handle to the created font.
     */
    HFONT setFont(WinHandle hWnd, long fontSize, BOOL isBold = FALSE, BOOL isItalic = FALSE, BOOL isUnderlined = FALSE, const wstring& fontName = L"Arial");

public:
    static COLORREF BackgroundColor;                   /**< General background color */
    static COLORREF ForegroundColor;                   /**< General foreground color */
    static COLORREF CaptionBackgroundStartColor;            /**< Frame Caption Bar start background color */
    static COLORREF CaptionBackgroundEndColor;            /**< Frame Caption Bar end background color */
    static COLORREF CaptionForegroundColor;            /**< Frame Caption Bar foreground color */
    static COLORREF ComponentBackgroundColor;          /**< Component background color */
    static COLORREF ComponentForegroundColor;          /**< Component foreground color */
    static COLORREF DisabledForegroundColor;            /**< Disabled text color */
    static COLORREF SelectionBackgroundColor;           /**< Selection background color */
    static COLORREF SelectionForegroundColor;           /**< Selection foreground color */
    static COLORREF GridAlternateBackgroundColor;
    static COLORREF GridAlternateForegroundColor;
    static COLORREF IconBackgroundColor;                   /**< Icon background color */
    static COLORREF IconForegroundColor;                   /**< Icon foreground color */

    static COLORREF ButtonBackgroundStartColor;         /**< Button background gradient start */
    static COLORREF ButtonBackgroundEndColor;           /**< Button background gradient end */
    static COLORREF ButtonForegroundColor;              /**< Button foreground color */

    static COLORREF ButtonPrimaryBackgroundStartColor;  /**< Primary button background gradient start */
    static COLORREF ButtonPrimaryBackgroundEndColor;    /**< Primary button background gradient end */
    static COLORREF ButtonPrimaryForegroundColor;       /**< Primary button foreground color */

    static COLORREF ButtonInfoBackgroundStartColor;  /**< Info button background gradient start */
    static COLORREF ButtonInfoBackgroundEndColor;    /**< Info button background gradient end */
    static COLORREF ButtonInfoForegroundColor;       /**< Info button foreground color */

    static COLORREF ButtonDangerBackgroundStartColor;   /**< Danger button background gradient start */
    static COLORREF ButtonDangerBackgroundEndColor;     /**< Danger button background gradient end */
    static COLORREF ButtonDangerForegroundColor;        /**< Danger button foreground color */

    static COLORREF ButtonSuccessBackgroundStartColor;  /**< Success button background gradient start */
    static COLORREF ButtonSuccessBackgroundEndColor;    /**< Success button background gradient end */
    static COLORREF ButtonSuccessForegroundColor;       /**< Success button foreground color */

    static COLORREF ButtonWarnBackgroundStartColor;  /**< Warn button background gradient start */
    static COLORREF ButtonWarnBackgroundEndColor;    /**< Warn button background gradient end */
    static COLORREF ButtonWarnForegroundColor;       /**< Warn button foreground color */

    static COLORREF ButtonGrayBackgroundStartColor;  /**< Gray button background gradient start */
    static COLORREF ButtonGrayBackgroundEndColor;    /**< Gray button background gradient end */
    static COLORREF ButtonGrayForegroundColor;       /**< Gray button foreground color */

    static COLORREF ButtonLightGrayBackgroundStartColor;  /**< LightGray button background gradient start */
    static COLORREF ButtonLightGrayBackgroundEndColor;    /**< LightGray button background gradient end */
    static COLORREF ButtonLightGrayForegroundColor;       /**< LightGray button foreground color */

    static COLORREF ButtonBlueBackgroundStartColor;  /**< Blue button background gradient start */
    static COLORREF ButtonBlueBackgroundEndColor;    /**< Blue button background gradient end */
    static COLORREF ButtonBlueForegroundColor;       /**< Blue button foreground color */

    static COLORREF ButtonLightBlueBackgroundStartColor;  /**< LightBlue button background gradient start */
    static COLORREF ButtonLightBlueBackgroundEndColor;    /**< LightBlue button background gradient end */
    static COLORREF ButtonLightBlueForegroundColor;       /**< LightBlue button foreground color */

    static COLORREF ButtonTealBackgroundStartColor;  /**< Teal button background gradient start */
    static COLORREF ButtonTealBackgroundEndColor;    /**< Teal button background gradient end */
    static COLORREF ButtonTealForegroundColor;       /**< Teal button foreground color */

    static COLORREF ButtonCyanBackgroundStartColor;  /**< Cyan button background gradient start */
    static COLORREF ButtonCyanBackgroundEndColor;    /**< Cyan button background gradient end */
    static COLORREF ButtonCyanForegroundColor;       /**< Cyan button foreground color */

    static COLORREF ButtonPinkBackgroundStartColor;  /**< Pink button background gradient start */
    static COLORREF ButtonPinkBackgroundEndColor;    /**< Pink button background gradient end */
    static COLORREF ButtonPinkForegroundColor;       /**< Pink button foreground color */

    static COLORREF ButtonIndigoBackgroundStartColor;  /**< Indigo button background gradient start */
    static COLORREF ButtonIndigoBackgroundEndColor;    /**< Indigo button background gradient end */
    static COLORREF ButtonIndigoForegroundColor;       /**< Indigo button foreground color */

    static COLORREF ButtonOrangeBackgroundStartColor;  /**< Orange button background gradient start */
    static COLORREF ButtonOrangeBackgroundEndColor;    /**< Orange button background gradient end */
    static COLORREF ButtonOrangeForegroundColor;       /**< Orange button foreground color */

    static COLORREF ButtonBrownBackgroundStartColor;  /**< Brown button background gradient start */
    static COLORREF ButtonBrownBackgroundEndColor;    /**< Brown button background gradient end */
    static COLORREF ButtonBrownForegroundColor;       /**< Brown button foreground color */

    static COLORREF ButtonSilverBackgroundStartColor;  /**< Silver button background gradient start */
    static COLORREF ButtonSilverBackgroundEndColor;    /**< Silver button background gradient end */
    static COLORREF ButtonSilverForegroundColor;       /**< Silver button foreground color */

    static COLORREF ButtonRedBackgroundStartColor;  /**< Red button background gradient start */
    static COLORREF ButtonRedBackgroundEndColor;    /**< Red button background gradient end */
    static COLORREF ButtonRedForegroundColor;       /**< Red button foreground color */

    static COLORREF HeaderBackgroundStartColor;         /**< Header background gradient start */
    static COLORREF HeaderBackgroundEndColor;           /**< Header background gradient end */
    static COLORREF HeaderForegroundColor;               /**< Header foreground color */

    static COLORREF ScrollbarTrackBackgroundColor;
    static COLORREF ScrollbarTrackForegroundColor;
    static COLORREF ScrollbarButtonBackgroundColor;
    static COLORREF ScrollbarButtonForegroundColor;

    static COLORREF LinkColor;                            /**< Color used for hyperlinks */
    static COLORREF LineColor;                            /**< Color used for lines and borders */
    static COLORREF DangerColor;                          /**< Color used for error/danger states */
    static COLORREF ShadowColor;                          /**< Color used for shadow background color */
};
