#pragma once

#include "Component.h"

class ToolbarItem
{
public:
    long _id;
    wstring _text;
    IconSource _iconSource;
    Direction _iconPosition;
    BOOL _isDisabled;
    Bounds _rect;
};


/**
 * @class Toolbar
 * @brief A UI component that displays a list of action items.
 */
class Toolbar : public Component
{
    /**
     * @brief The list of items currently in the toolbar.
     */
    vector<ToolbarItem> _items;

    long _firstVisibleItem;              /**< Index of the first visible item (scrolling). */
    long _lastVisibleItem;               /**< Index of the last visible item. */

    long _indexOfPressedItem;            /**< Index of the currently pressed item. */
    BOOL _lastItemIsPartiallyDisplayed;

    Bounds _scrollLeftBtnRect, _scrollRightBtnRect;

protected:
    /**
     * @brief Retrieves the options for creating the toolbar window.
     *
     * @param title Output: window title.
     * @param widownStyles Output: window style flags.
     * @param wndClassName Output: window class name.
     * @param isCustomWndProc Output: whether a custom window procedure is used.
     * @return TRUE if the options were successfully retrieved, FALSE otherwise.
     */
    BOOL getCreateWindowOptions(wstring& title, UINT& widownStyles, wstring& wndClassName, BOOL& isCustomWndProc);

    /**
     * @brief Called after the toolbar window is created.
     */
    void windowCreated();

public:
    /**
     * @brief Constructs a new Toolbar instance.
     */
    Toolbar();

    /**
     * @brief Destroys the Toolbar instance.
     */
    virtual ~Toolbar();

    /**
     * @brief Called when the toolbar window is resized.
     */
    void onWindowResized();

    /**
     * @brief Paints the toolbar on the screen.
     * @param hDC Handle to the device context.
     */
    void onPaint(Graphics *g);

    /** @name Mouse Events */
    ///@{
    void onMousePressed(WinHandle hWnd, int x, int y, int clickCount, BOOL shiftPressed, BOOL ctrlPressed);
    void onMouseReleased(WinHandle hWnd, int x, int y, BOOL shiftPressed, BOOL ctrlPressed);
    BOOL onMouseWheel(WinHandle hWnd, int x, int y, int delta);
    ///@}

    /**
     * @brief Adds an item to the toolbar.
     *
     * @param id id of the item.
     * @param text The text displayed on the item in the toolbar.
     * @param iconFile The path of the image file to be displayed.
     * @param iconPosition The position of the image to be displayed.
     */
    void addItem(long id, const wstring& text, const IconSource& iconSource = IconSource(), const Direction& iconPosition = Direction::LEFT);

    /**
     * @brief enables/disabled items.
     *
     * @param id id of the item.
     * @param enabled TRUE to enable, FALSE to disable the item.
     */
    void enableItem(long id, BOOL enabled = TRUE);

    /**
     * @brief Handles an action event for this component.
     *
     * @param hTarget Handle to the control that triggered the action.
     * @param actionId The ID of the action.
     * @param procParams Additional parameters for the action.
     */
    void onAction(WinHandle hTarget, long actionId, const ProcParams& procParams);

    /**
     * @brief Adds an action listener to the button to handle action events.
     *
     * @param l Pointer to the ActionListener object to add.
     */
    void addActionListener(ActionListener* l);
};
